/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.utils;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonFactoryBuilder;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.json.JsonWriteFeature;
import com.sap.cds.services.messages.Message;
import com.sap.cds.services.messages.MessageTarget;
import com.sap.cds.services.messages.Messages;
import com.sap.cds.services.utils.ODataUtils;
import com.sap.cds.services.utils.StringUtils;

public class MessagesUtils {

	private static final Logger logger = LoggerFactory.getLogger(MessagesUtils.class);
	private static final String CODE = "code";
	private static final String MESSAGE = "message";
	private static final String NUMERIC_SEVERITY = "numericSeverity";
	private static final String TARGET = "target";
	private static final String LONG_TEXT_URL = "longtextUrl";

	public static String getSapMessagesHeader(String bindingParameter, Messages messages) {
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
		JsonFactoryBuilder builder = new JsonFactoryBuilder().configure(JsonWriteFeature.ESCAPE_NON_ASCII, true);
		try (JsonGenerator json = new JsonFactory(builder).createGenerator(outputStream)) {
			boolean isFirst = true;
			Iterator<Message> iterator = messages.stream().iterator();
			while (iterator.hasNext()) {
				if (isFirst) {
					json.writeStartArray();
					isFirst = false;
				}
				Message message = iterator.next();
				json.writeStartObject();
				// mandatory
				json.writeStringField(CODE, getMessageCode(message));
				json.writeStringField(MESSAGE, message.getMessage());
				json.writeNumberField(NUMERIC_SEVERITY, message.getSeverity().getNumericSeverity());
				// optional
				String target = getTarget(bindingParameter, message.getTarget());
				if(target != null) {
					json.writeStringField(TARGET, target);
				}
				String[] stringifiedTargets = message.getAdditionalTargets().stream()
					.map(t -> MessagesUtils.getTarget(bindingParameter, t))
					.filter(Objects::nonNull).toArray(i -> new String[i]);
				if (stringifiedTargets.length > 0) {
					json.writeFieldName("additionalTargets");
					json.writeArray(stringifiedTargets, 0, stringifiedTargets.length);
				}
				if(!StringUtils.isEmpty(message.getLongTextUrl())) {
					json.writeStringField(LONG_TEXT_URL, message.getLongTextUrl());
				}
				json.writeEndObject();
			}
			if (!isFirst) {
				json.writeEndArray();
			}
			json.close();
			if (outputStream.size() != 0) {
				return outputStream.toString("UTF-8");
			}
		} catch (IOException e) {
			logger.error("Failed to create sap messages header", e);
		}
		return null;
	}

	public static String getMessageCode(Message message) {
		return StringUtils.isEmpty(message.getCode()) ? "<none>" : message.getCode();
	}

	public static String getTarget(String bindingParameter, MessageTarget messageTarget) {
		if (messageTarget == null) {
			return null;
		}
		return ODataUtils.toODataTarget(bindingParameter, messageTarget.getParameter(), messageTarget.getRef(), false);
	}

}
