/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.adapter.odata.v4.utils;

import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZonedDateTime;
import java.util.GregorianCalendar;
import java.util.UUID;

import org.apache.olingo.commons.api.edm.EdmEnumType;
import org.apache.olingo.commons.api.edm.EdmPrimitiveType;
import org.apache.olingo.commons.api.edm.EdmPrimitiveTypeException;
import org.apache.olingo.commons.api.edm.EdmType;
import org.apache.olingo.commons.api.edm.EdmTypeDefinition;
import org.apache.olingo.commons.api.edm.constants.EdmTypeKind;
import org.apache.olingo.commons.core.edm.primitivetype.EdmBinary;
import org.apache.olingo.commons.core.edm.primitivetype.EdmBoolean;
import org.apache.olingo.commons.core.edm.primitivetype.EdmByte;
import org.apache.olingo.commons.core.edm.primitivetype.EdmDate;
import org.apache.olingo.commons.core.edm.primitivetype.EdmDateTimeOffset;
import org.apache.olingo.commons.core.edm.primitivetype.EdmDecimal;
import org.apache.olingo.commons.core.edm.primitivetype.EdmDouble;
import org.apache.olingo.commons.core.edm.primitivetype.EdmGuid;
import org.apache.olingo.commons.core.edm.primitivetype.EdmInt16;
import org.apache.olingo.commons.core.edm.primitivetype.EdmInt32;
import org.apache.olingo.commons.core.edm.primitivetype.EdmInt64;
import org.apache.olingo.commons.core.edm.primitivetype.EdmSByte;
import org.apache.olingo.commons.core.edm.primitivetype.EdmSingle;
import org.apache.olingo.commons.core.edm.primitivetype.EdmString;
import org.apache.olingo.commons.core.edm.primitivetype.EdmTimeOfDay;

import com.sap.cds.util.CdsTypeUtils;


/**
 * Utils to map Olingo types to CAP Java types and vice versa.
 */
public class TypeConverterUtils {

	private TypeConverterUtils() {

	}

	/**
	 * Converts a String value to the correct Java value, based on the EdmType
	 * @param type the EdmType
	 * @param value the string value
	 * @return the value in the correct Java type
	 */
	public static Object convertToType(EdmType type, String value) {
		if(value == null) {
			return null;
		}

		Object typedValue;

		EdmType theType;
		if(type.getKind() == EdmTypeKind.ENUM) {
			theType = ((EdmEnumType) type).getUnderlyingType();
		} else if (type.getKind() == EdmTypeKind.DEFINITION) {
			theType = ((EdmTypeDefinition) type).getUnderlyingType();
		} else {
			theType = type;
		}

		if (theType instanceof EdmPrimitiveType primitiveType) {
			try {
				value = primitiveType.fromUriLiteral(value);
			} catch (EdmPrimitiveTypeException e) {
				//ignore, value is already in correct format
			}
		}

		typedValue = convertPrimitiveType(theType, value);

		return typedValue;
	}

	private static Object convertPrimitiveType(EdmType type, String value) {
		Object typedValue;
		if (type instanceof EdmDecimal) {
			typedValue = new BigDecimal(value);
		} else if (type instanceof EdmDouble) {
			typedValue = Double.parseDouble(value);
		} else if (type instanceof EdmSingle) {
			typedValue = Float.parseFloat(value);
		} else if (type instanceof EdmInt64) {
			typedValue = Long.parseLong(value);
		} else if (type instanceof EdmInt32 ||
			type instanceof EdmInt16 ||
			type instanceof EdmSByte ||
			type instanceof EdmByte) {
			typedValue = Integer.parseInt(value);
		} else if (type instanceof EdmBinary) {
			typedValue = value.getBytes(Charset.defaultCharset());
		} else if (type instanceof EdmString) {
			typedValue = value;
		} else if (type instanceof EdmTimeOfDay) {
			typedValue = LocalTime.parse(value);
		} else if (type instanceof EdmDate) {
			typedValue = LocalDate.parse(value);
		} else if (type instanceof EdmDateTimeOffset) {
			typedValue = ZonedDateTime.parse(value).toInstant();
		} else if (type instanceof EdmBoolean) {
			typedValue = Boolean.parseBoolean(value);
		} else if (type instanceof EdmGuid) {
			typedValue = CdsTypeUtils.parseUuid(value);
		} else {
			typedValue = value;
		}
		return typedValue;
	}

	/**
	 * This method converts the result from db to correct Java Type required by Olingo serializers
	 * for types Guid, DateTime, Date and Time
	 * @param type EdmType
	 * @param value Value
	 * @return java Object
	 */
	public static Object getValueBasedOnTypeOfResultSet(final EdmType type, Object value) {
		if (value == null) {
			return null;
		}
		if (type instanceof EdmGuid) {
			value = UUID.fromString(value.toString());
		} else if (value instanceof String string &&
			(type instanceof EdmDateTimeOffset || type instanceof EdmDate || type instanceof EdmTimeOfDay)) {

			value = TypeConverterUtils.convertToType(type, string);
		}

		return value;
	}

	/**
	 * This method converts input payload to correct Java Types for types Guid, DateTime, Date and Time
	 * The handlers will get the new java Date Time types because of this conversion
	 * @param typeName edmTypeName
	 * @param value Value
	 * @return java Object
	 */
	public static Object getValueBasedOnTypeOfRequestPayload(final String typeName, Object value) {
		if (typeName != null) {
			switch (typeName) {
				case "Edm.Guid":
					if (value instanceof UUID) {
						value = value.toString();
					}
					break;
				case "Edm.DateTimeOffset":
					if (value instanceof java.sql.Timestamp timestamp) {
						value = timestamp.toInstant();
					}
					break;
				case "Edm.Date":
					if (value instanceof GregorianCalendar calendar) {
						value = calendar.toZonedDateTime().toLocalDate();
					}
					break;
				case "Edm.TimeOfDay":
					if (value instanceof GregorianCalendar calendar) {
						value = calendar.toZonedDateTime().toLocalTime();
					}
					break;
				default:
					return value;
			}
		}
		return value;
	}
}
