/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature;

import java.util.Collections;
import java.util.List;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility class to load features with fallback default implementations.
 * Features can be in active or inactive state.
 * The first feature that is found in active state is used.
 * If no active feature is found the default feature is used instead.
 */
public class FeatureLoader {

	private static final Logger logger = LoggerFactory.getLogger(FeatureLoader.class);

	private static Set<Feature> loadedFeatures = ConcurrentHashMap.newKeySet();

	/**
	 * Loads a single feature for the interface and a default one, if none is available
	 *
	 * @param <T> the type of the feature
	 * @param featureInterface the feature interface
	 * @param defaultFeature the supplier for the default feature
	 * @return the feature instance
	 */
	public static <T extends Feature> T getFeature(Class<T> featureInterface, Supplier<T> defaultFeature) {

		T feature = loadFeatures(featureInterface).filter(Feature::isActiveFeature).findFirst().orElse(null);

		if(feature == null && defaultFeature != null) {
			feature = defaultFeature.get();
		}

		if(feature != null) {
			logger.info("Loaded feature '{}'", feature.getFeatureName());
			loadedFeatures.add(feature);
		}
		return feature;
	}

	/**
	 * Loads a list of features for the interface and a default one, if none is available
	 *
	 * @param <T> the type of the feature
	 * @param featureInterface the feature interface
	 * @param defaultFeature the supplier for the default feature
	 * @return the feature instance list
	 */
	public static <T extends Feature> List<T> getFeatureList(Class<T> featureInterface, Supplier<T> defaultFeature) {

		List<T> featureList = loadFeatures(featureInterface).filter(Feature::isActiveFeature).collect(Collectors.toList());

		if(featureList.isEmpty() && defaultFeature != null) {
			featureList.add(defaultFeature.get());
		}

		for(T feature : featureList) {
			logger.info("Loaded feature '{}'", feature.getFeatureName());
			loadedFeatures.add(feature);
		}

		return Collections.unmodifiableList(featureList);
	}

	/**
	 * @return A {@link Stream} of loaded features
	 */
	public static Stream<Feature> loadedFeatures() {
		return loadedFeatures.stream();
	}

	/**
	 * Prepares a {@link Stream} of {@link Feature}s with the given interface.
	 *
	 * @param featureInterface	The interface to check
	 * @return	The {@link Stream} of {@link Feature}s
	 */
	private static <T extends Feature> Stream<T> loadFeatures(Class<T> featureInterface) {
		ServiceLoader<T> serviceLoader = ServiceLoader.load(featureInterface);
		Iterable<T> iterable = () -> serviceLoader.iterator();
		return StreamSupport.stream(iterable.spliterator(), false);
	}
}
