/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.config;

import com.sap.cds.feature.Feature;

/**
 * The interface exposing configuration properties.
 */
public interface PropertiesProvider extends Feature {

	/**
	 * Binds the configuration POJO to the specified section
	 * @param <T> the type of the POJO
	 * @param section the properties section
	 * @param clazz the class of the POJO
	 *
	 * @return The properties instance
	 */
	<T> T bindPropertyClass(String section, Class<T> clazz);

	/**
	 * Checks if the property with given key is defined.
	 *
	 * @param key	The key of the property
	 * @return {@code true} if the property with key {@code key} is defined
	 */
	default boolean	hasProperty(String key) {
		return getProperty(key) != null;
	}

	/**
	 * Returns the value of the property value with given key as {@code String}.
	 *
	 * @param key	The key of the property
	 * @return	The {@code String} value or {@code null} if not defined
	 */
	default String getProperty(String key) {
		return getProperty(key, (String) null);
	}

	/**
	 * Returns the value of the property with given key as {@code String}.
	 * Returns the specified default value if key is not existing.
	 *
	 * @param key	The key of the property
	 * @param defaultValue The default value if the key is not present
	 * @return	The {@code String} value or {@code defaultValue} if not defined
	 */
	String getProperty(String key, String defaultValue);

	/**
	 * Returns the value of the property with given key as given type {@code asClazz}.
	 *
	 * @param <T> the type of the property value
	 * @param key	The key of the property
	 * @param asClazz The type of the property value
	 * @return	The value or null if not defined
	 */
	default <T> T getProperty(String key, Class<T> asClazz) {
		return getProperty(key, asClazz, (T) null);
	}

	/**
	 * Returns the value of the property with given key as given type {@code asClazz}.
	 * Returns the specified default value if key is not existing.
	 *
	 * @param <T> the type of the property value
	 * @param key	The key of the property
	 * @param asClazz The type of the property value
	 * @param defaultValue The default value if the key is not present
	 * @return	The property value or {@code defaultValue} if not defined
	 */
	<T> T getProperty(String key, Class<T> asClazz, T defaultValue);

}
