/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.config.local;

import java.io.InputStream;
import java.lang.reflect.Method;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;

import org.yaml.snakeyaml.TypeDescription;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.Constructor;
import org.yaml.snakeyaml.nodes.Node;
import org.yaml.snakeyaml.nodes.ScalarNode;
import org.yaml.snakeyaml.representer.Representer;

import com.sap.cds.feature.config.PropertiesProvider;
import com.sap.cds.feature.config.pojo.CdsProperties;
import com.sap.cds.feature.config.pojo.PlainProperties;

/**
 * Default implementation of {@link PropertiesProvider} based on parsing a yaml-config file.
 * <p>
 * Per default, the properties are loaded from ./application.yaml in the resource path.
 * {@code -DconfigFile=/somePath/config.yaml} overrides the default location.
 * <p>
 * Per default, the active profile is either the first profile with name 'default' or the first profile in the yaml.
 * {@code -DconfigProfile=<profile-name>} switches the active profile.
 */
public class DefaultPropertiesProvider implements PropertiesProvider {

	/**
	 * The path to the configuration file
	 */
	private final String configFile = System.getProperty("configFile", "application.yaml"); // don't use static here!

	/**
	 * The name of the configuration profile. If "default" is not set, the first profile will be taken.
	 */
	private final String configProfile = System.getProperty("configProfile", "default"); // don't use static here!

	/**
	 * The loaded properties as {@link PropertiesRoot}
	 */
	private PropertiesRoot root;

	/**
	 * The loaded normalized YAML properties of the active profile as Map
	 */
	private Map<String, Object> yamlMap = new HashMap<>();

	/**
	 * The loaded normalized System properties as Map
	 */
	private Map<String, Object> systemPropsMap = new HashMap<>();

	/**
	 * The loaded normalized Environment variables as Map
	 */
	private Map<String, Object> envMap = new HashMap<>();

	public DefaultPropertiesProvider() {
		// order is important here
		// highest priority last
		loadYaml();
		loadEnvironmentVariables();
		loadSystemProperties();
	}

	/**
	 * Loads properties from the YAML configFile into the {@link PropertiesRoot} structure.
	 * Also loads properties from the YAML configFile into the YAML Map
	 */
	private void loadYaml() {
		int position = -1;

		// read POJO
		InputStream pojoInputStream = DefaultPropertiesProvider.class.getClassLoader().getResourceAsStream(configFile);
		if(pojoInputStream != null) {
			Representer representer = new Representer();
			representer.getPropertyUtils().setSkipMissingProperties(true);

			Yaml yaml = new Yaml(new CustomConstructor(PropertiesRoot.class), representer);
			for (Object object : yaml.loadAll(pojoInputStream)) {
				++position;
				if (object instanceof PropertiesRoot) {
					PropertiesRoot properties = (PropertiesRoot) object;
					String[] profiles = properties.getPlain().getProfiles().split(",");
					if (Arrays.asList(profiles).stream().map(p -> p.trim()).anyMatch(p -> p.equals(configProfile))) {
						// TODO if we want to support multiple active profiles
						// we need some merging strategy here and for the map below
						root = properties; // first found wins
						break;
					}
				}
			}
		}

		// read map
		if(root != null && position >= 0) {
			InputStream mapInputStream = DefaultPropertiesProvider.class.getClassLoader().getResourceAsStream(configFile);
			if(mapInputStream != null) {
				Yaml yaml = new Yaml();
				int p = -1;
				for(Object object : yaml.loadAll(mapInputStream)) {
					if(position == ++p && object instanceof Map) {
						writeToYamlMap(null, (Map<?, ?>) object);
					}
				}
			}
		}

		// default values
		if(root == null) {
			root = new PropertiesRoot();
		}
	}

	private static class CustomConstructor extends Constructor {

		public CustomConstructor(Class<? extends Object> theRoot) {
			super(theRoot);
			typeDefinitions.put(Duration.class, new DurationTypeDescription());
		}

		private static class DurationTypeDescription extends TypeDescription {
			public DurationTypeDescription() {
				super(Duration.class);
			}

			@Override
			public Object newInstance(Node node) {
				ScalarNode snode = (ScalarNode) node;
				// TODO: This parse method only supports ISO-8601 format, not simple values like 15s
				return Duration.parse(snode.getValue());
			}
		}

	}

	/**
	 * Flattens the given map and adds objects under their full normalized key to the YAML map
	 * @param prefix the current prefix to the next key-level
	 * @param map the currently processed Map
	 */
	private void writeToYamlMap(String prefix, Map<?, ?> map) {
		for(Entry<?, ?> entry : map.entrySet()) {
			String key;
			if(prefix == null) {
				key = entry.getKey().toString();
			} else {
				key = prefix + "." + entry.getKey().toString();
			}

			if(entry.getValue() instanceof Map) {
				writeToYamlMap(key, (Map<?, ?>) entry.getValue());
			} else if (entry.getValue() instanceof Iterable) {
				// TODO support handling lists in keys
			} else {
				yamlMap.put(normalizeKey(key), entry.getValue());
			}
		}
	}

	/**
	 * Iterates over all system properties and writes matching configuration properties to the root object
	 */
	private void loadSystemProperties() {
		for(Entry<Object, Object> entry: System.getProperties().entrySet()) {
			if(entry.getKey() instanceof String && entry.getValue() instanceof String) {
				handleProperty(root, (String) entry.getKey(), (String) entry.getValue());
				systemPropsMap.put(normalizeKey((String) entry.getKey()), entry.getValue());
			}
		}
	}

	/**
	 * Iterates over all environment variables and writes matching configuration properties to the root object
	 */
	private void loadEnvironmentVariables() {
		for(Entry<String, String> entry : System.getenv().entrySet()) {
			handleProperty(root, entry.getKey(), entry.getValue());
			envMap.put(normalizeKey(entry.getKey()), entry.getValue());
		}
	}

	/**
	 * Handles properties coming from string based key value pairs.
	 * These properties are normalized and dots in the key are treated as separators.
	 * The first key parts are used to call a hierarchy of getters on the given root object.
	 * The last key part is used to call a setter on the final object.
	 *
	 * @param root the root object
	 * @param propertyKey the property key
	 * @param propertyValue the property value
	 */
	private void handleProperty(Object root, String propertyKey, String propertyValue) {
		Object obj = root;
		String key = normalizeKey(propertyKey);
		String[] keyParts = key.split("\\.");
		for(int i = 0; i < keyParts.length; i++) {
			// TODO support handling lists indexes in keyParts
			String keyPart = keyParts[i];
			// look for getters on non-last parts
			boolean getter = i < (keyParts.length - 1);
			Method method = findGetterOrSetterMethod(obj.getClass(), getter, keyPart);

			// no method found, property is not part of known properties
			if(method == null) {
				return;
			}

			try {
				if(getter) {
					obj = method.invoke(obj);
				} else {
					method.invoke(obj, convertValue(propertyValue, method.getParameters()[0].getType()));
				}
			} catch (Exception e) {
				throw new RuntimeException("Failed to handle property " + propertyKey, e); // NOSONAR
			}
		}
	}

	/**
	 * Normalizes a key, by removing all "-" and replacing "_" with "."
	 * In addition the key is converted to lower case.
	 *
	 * @param key the unnormalized key
	 * @return the normalized key
	 */
	private String normalizeKey(String key) {
		return key.replace("-", "").replace("_", ".").toLowerCase(Locale.ENGLISH);
	}

	/**
	 * Finds a getter or setter on the given class, that corresponds to the specified attribute name
	 * Can handle getters starting with get or is
	 *
	 * @param clazz the class to analyze
	 * @param getter true, if a getter or false if a setter should be returned
	 * @param name the name of the attribute (in lowercase)
	 * @return the found method or <code>null</code>
	 */
	private Method findGetterOrSetterMethod(Class<?> clazz, boolean getter, String name) {
		for(Method method : clazz.getMethods()) {
			String methodName = method.getName().toLowerCase(Locale.ENGLISH);
			if(getter) {
				if((methodName.startsWith("get") && methodName.equals("get" + name)) || (methodName.startsWith("is") && methodName.equals("is" + name))) {
					if(method.getParameterCount() == 0 && !method.getReturnType().equals(Void.TYPE)) {
						return method;
					}
				}
			} else {
				if(methodName.startsWith("set") && methodName.equals("set" + name)) {
					if (method.getParameterCount() == 1 && method.getReturnType().equals(Void.TYPE)) {
						return method;
					}
				}
			}
		}
		return null;
	}

	/**
	 * Converts a given string value to the specified type
	 * @param value the string value
	 * @param type the type to convert the value to
	 * @return the converted value
	 */
	@SuppressWarnings("unchecked")
	private <T> T convertValue(String value, Class<T> type) {
		if (type.equals(String.class)) {
			return (T) value;
		} else if (type.equals(Integer.class) || type.equals(Integer.TYPE)) {
			return (T) (Integer) Integer.parseInt(value);
		} else if (type.equals(Long.class) || type.equals(Long.TYPE)) {
			return (T) (Long) Long.parseLong(value);
		} else if (type.equals(Boolean.class) || type.equals(Boolean.TYPE)) {
			return (T) (Boolean) Boolean.parseBoolean(value);
		} else if (type.equals(Duration.class)) {
			// TODO: This parse method only supports ISO-8601 format, not simple values like 15s
			return (T) Duration.parse(value);
		} else {
			throw new UnsupportedOperationException("Type " + type + " not supported");
		}
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T> T bindPropertyClass(String section, Class<T> clazz) {
		if("cds".equals(section) && CdsProperties.class.equals(clazz)) {
			return (T) root.getCds();
		} else if("plain".equals(section) && PlainProperties.class.equals(clazz)) {
			return (T) root.getPlain();
		}
		throw new UnsupportedOperationException("Currently only CdsProperties and PlainProperties are supported");
	}

	@Override
	public String getProperty(String key, String defaultValue) {
		String normalizedKey = normalizeKey(key);
		Object value = systemPropsMap.get(normalizedKey);
		if(value == null) {
			value = envMap.get(normalizedKey);
		}
		if(value == null) {
			value = yamlMap.get(normalizedKey);
		}
		return value == null ? defaultValue : value.toString();
	}

	@Override
	public <T> T getProperty(String key, Class<T> clazz, T defaultValue) {
		String value = getProperty(key);
		return value == null ? defaultValue : convertValue(value, clazz);
	}

	@Override
	public boolean isActiveFeature() {
		return true;
	}

	@Override
	public String getFeatureName() {
		return "Default Properties Provider";
	}

	/**
	 * The YAML root representation
	 */
	public static class PropertiesRoot {

		CdsProperties cds = new CdsProperties();
		PlainProperties plain = new PlainProperties();

		public CdsProperties getCds() {
			return cds;
		}

		public void setCds(CdsProperties cds) {
			this.cds = cds;
		}

		public PlainProperties getPlain() {
			return plain;
		}

		public void setPlain(PlainProperties plain) {
			this.plain = plain;
		}

	}

}
