/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.config.pojo;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.docs.config.DocumentedProperty;
import com.sap.cds.feature.config.pojo.CdsProperties.Application.ApplicationServiceConfig;
import com.sap.cds.feature.config.pojo.CdsProperties.Composite.CompositeServiceConfig.Route;

/**
 * Properties that control the CDS runtime
 */
@lombok.Getter
@lombok.Setter
public class CdsProperties {

	/** Properties for environments, like local development or CloudFoundry. */
	private Environment environment = new Environment();
	/** Properties for the primary data source used by the default persistence service. */
	private DataSource dataSource = new DataSource();
	/** Properties for the CDS model. */
	private Model model = new Model();
	/** Properties for security configurations of services and endpoints. */
	private Security security = new Security();
	/** Properties for the index page. */
	private Servlet indexPage = new Servlet("/");
	/** Properties for the OData V4 protocol adapter. */
	private ODataV4 odataV4 = new ODataV4();
	/** Properties for the OData V2 protocol adapter. */
	private ODataV2 odataV2 = new ODataV2();
	/** Properties for messaging services. */
	private Messaging messaging = new Messaging();
	/** Properties for multitenancy and extensibility. */
	private MultiTenancy multiTenancy = new MultiTenancy();
	/** Properties for application services. */
	private Application application = new Application();
	/** Properties for remote services. */
	private Remote remote = new Remote();
	/** Properties for locale configurations. */
	private Locales locales = new Locales();
	/** Properties for error handling. */
	private Errors errors = new Errors();
	/** Properties for draft-enabled entities. */
	private Drafts drafts = new Drafts();
	/** Properties for augmentation of CQN queries. */
	private Query query = new Query();

	@lombok.Getter
	@lombok.Setter
	public static class Environment {

		/** Properties for the local environment. */
		private Local local = new Local();
		/** Properties for Kubernetes. */
		private K8s k8s = new K8s();

		@lombok.Getter
		@lombok.Setter
		public static class Local {

			/**
			 * File-system path to the default environment JSON file.
			 * The file follows the structure of CloudFoundry's VCAP_SERVICES and VCAP_APPLICATION environment variables.
			 * If this property specifies a folder, the filename `default-env.json` is appended to it.
			 */
			private String defaultEnvPath;

		}

		@lombok.Getter
		@lombok.Setter
		public static class K8s {
			/**
			 * Secrets directory where service bindings are found according to the following conventions:
			 * `[secretsPath]/[serviceName]/[serviceInstanceName]`
			 * `[serviceName]` and `[serviceInstanceName]` are directories with the name of the service and service instance respectively.
			 * The directory `[serviceInstanceName]` must contain the service binding credentials as files with its name being the key and its content being the value.
			 */
			private String secretsPath = "/etc/secrets/sapcp";
			/**
			 * Properties to explicitly configure service bindings with an arbitrary secrets path.
			 * The metadata for these service bindings can be fully specified, to ensure CAP is able detect the binding in auto-configuration.
			 * The key can be chosen arbitrarily and is used as the service binding name, if the name property is not explicitly defined.
			 */
			private Map<String, ServiceBindingConfig> serviceBindings = new HashMap<>();

			public Map<String, ServiceBindingConfig> getServiceBindings() {
				serviceBindings.forEach((k, v) -> {
					if(v.getName() == null || v.getName().trim().isEmpty()) {
						v.setName(k);
					}
				});
				return serviceBindings;
			}

			@lombok.Getter
			@lombok.Setter
			public static class ServiceBindingConfig {
				/** The name of the service binding. */
				private String name;
				/** The path to the secrets directory. */
				private String secretsPath;
				/** The name of the service. */
				private String service;
				/** The optional name of the service plan. */
				private String plan;
				/** The optional list of service tags. */
				private List<String> tags;
			}
		}
	}

	@lombok.Getter
	@lombok.Setter
	public static class DataSource {

		/** Determines, if the data source is considered embedded (in-memory). */
		private boolean embedded;
		/** The name of the primary service binding, used by the default persistence service. */
		private String binding;
		/**
		 * Determines in which scenarios the default persistence service is initialized with CSV data.
		 * By default CSV initialization only happens, if the data source is embedded (in-memory).
		 * Possible values are: `embedded`, `never`, `always`.
		 */
		private String csvInitializationMode = "embedded"; // or 'never' or 'always'
		/** The file suffix of CSV files. */
		private String csvFileSuffix = ".csv";
		/**
		 * The file-system paths to search for CSV files in.
		 * Using `/**` at the end of the path triggers a recursive search.
		 */
		private List<String> csvPaths = Arrays.asList("db/data/**", "db/csv/**", "../db/data/**", "../db/csv/**");

		// BACKWARDS COMPATIBILITY: cds.dataSource.serviceName was renamed to cds.dataSource.binding

		@Deprecated
		public String getServiceName() {
			return binding;
		}

		@Deprecated
		public void setServiceName(String binding) {
			this.binding = binding;
		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Model {

		/** The resource path to the csn.json file. */
		private String csnPath = "edmx/csn.json";

	}

	@lombok.Getter
	@lombok.Setter
	public static class Security {

		/**
		 * Determines, if protocol adapter endpoints that do not require authorization also do not enforce authentication.
		 * If not set explicitly, a reasonable default value is derived the from the application scenario:
		 * In the multitenant scenario, which usually requires authentication to obtain the tenant information, the default value is `false`.
		 * For all other scenarios, `true` is the default value.
		 */
		private Boolean openUnrestrictedEndpoints;
		/** Determines, if OData $metadata endpoints are available without authentication. */
		private boolean openMetadataEndpoints = false;
		/** Determines, if security configurations enforce authentication for endpoints not managed by CAP protocol-adapters. */
		private boolean authenticateUnknownEndpoints = true;

		/**
		 * Properties to control the protection of drafts.
		 * If a draft is protected it is only accessible by its creator.
		 */
		private Enabled draftProtection = new Enabled(true);
		/** Properties for instance-based authorization checks defined in @restrict annotations. */
		@DocumentedProperty(false)
		private Enabled instanceBasedAuthorization = new Enabled(true);
		/** Properties for deriving authorization of autoexposed entities from the request path. */
		@DocumentedProperty(false)
		private Enabled authorizeAutoExposedEntities = new Enabled(true);

		/** Properties for XSUAA based authentication. */
		private Xsuaa xsuaa = new Xsuaa();
		/** Properties for mock-user based authentication. */
		private Mock mock = new Mock();

		/**
		 * Returns the effective value of property {@code openUnrestrictedEndpoints}.
		 *
		 * @param mt	{@code true} if the application runs in multitenant mode
		 * @return	the effective value of {@code openUnrestrictedEndpoints}
		 */
		public boolean getOpenUnrestrictedEndpoints(boolean mt) {
			Boolean result = openUnrestrictedEndpoints;
			if (result == null) {
				// ST -> we open unrestricted endpoints by default
				// MT -> we don't open unrestricted endpoints by default as we need the tenant
				result = !mt;
			}
			return result;
		}

		@lombok.Getter
		@lombok.Setter
		public static class Xsuaa extends Enabled {

			/** Properties for the XSUAA Spring security auto-configuration. */
			private Enabled authConfig = new Enabled(true);
			/** Determines, if user names from XSUAA tokens are normalized. */
			@DocumentedProperty(false)
			private boolean normalizeUserNames = false;
			/** The name of the XSUAA service binding, used for the XSUAA security auto-configuration. */
			private String binding;

			public Xsuaa() {
				super(true);
			}

			// BACKWARDS COMPATIBILITY: cds.security.xsuaa.serviceName was renamed to cds.security.xsuaa.binding

			@Deprecated
			public String getServiceName() {
				return binding;
			}

			@Deprecated
			public void setServiceName(String binding) {
				this.binding = binding;
			}

		}

		@lombok.Getter
		@lombok.Setter
		public static class Mock extends Enabled {

			/** The list of mock-users, used for basic authentication in local development and test scenarios. */
			private List<User> users = new ArrayList<>();

			public Mock() {
				super(true);
			}

			@lombok.Getter
			@lombok.Setter
			public static class User {

				/** The ID of the mock-user. */
				private String id;
				/**
				 * The (mandatory) name of the mock-user.
				 * It is used to perform the basic authentication.
				 */
				private String name; // mandatory
				/**
				 * The (optional) password of the mock-user.
				 * It is used to perform the basic authentication.
				 */
				private String password = ""; // empty password is default
				/** The tenant of the mock-user. */
				private String tenant; // optional
				/** Determines, if this mock-user is treated as a system user. */
				private boolean systemUser = false;
				/** Determines, if this mock-user is treated as the privileged user. */
				private boolean privileged = false;
				/** The list of roles, that are assigned to this mock-user. */
				private List<String> roles = new ArrayList<>();
				/**
				 * A map of user attributes, that are assigned to the mock-user.
				 * The name of the attribute needs to be given as the key.
				 * The attribute values are provided as a list.
				 */
				private Map<String, List<String>> attributes = new HashMap<>();
				/**
				 * A list of attribute names, for which the mock-user has no restrictions.
				 * This is treated as if the mock-user effectively had all possible values of this attribute assigned.
				 */
				private List<String> unrestricted = new ArrayList<>();
				/**
				 * A map of additional properties of the mock-user.
				 * It can be used to mock authentication specific properties (e.g. email address).
				 * The name of the additional attribute needs to be given as the key.
				 * The value of the attribute can be provided as an arbitrary object.
				 */
				private Map<String, Object> additional = new HashMap<>();

				public boolean isValid() {
					return getName() != null && !getName().isEmpty() && getPassword() != null;
				}

				@Override
				public String toString() {
					try {
						ObjectMapper mapper = new ObjectMapper();
						mapper.setSerializationInclusion(Include.NON_NULL);
						return mapper.writeValueAsString(this);
					} catch(Exception e) { // NOSONAR
						return name;
					}
				}

			}

		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class ODataV4 {

		/** Properties of the OData V4 protocol adapter endpoint. */
		private Servlet endpoint = new Servlet("/odata/v4");
		/** Properties of the OData V4 index page. */
		@Deprecated
		private Servlet indexPage = new Servlet("/");
		/** Determines, if URLs in the @odata.context response annotation are absolute. */
		private boolean contextAbsoluteUrl = false;
		/** The JAR resource path to search for OData V4 EDMX files. */
		private String edmxPath = "edmx/v4";

	}

	@lombok.Getter
	@lombok.Setter
	public static class ODataV2 {

		/** Properties of the OData V2 protocol adapter endpoint. */
		private Servlet endpoint = new Servlet("/odata/v2");
		/** The JAR resource path to search for OData V2 EDMX files. */
		private String edmxPath = "edmx/v2";

	}

	@lombok.Getter
	@lombok.Setter
	public static class Composite {

		/**
		 * Properties for composite services.
		 * The key can be chosen arbitrarily and is used as the composite service name, if the name property is not explicitly defined.
		 * In addition it can leveraged to split configuration across multiple profiles.
		 */
		private Map<String, CompositeServiceConfig> services = new HashMap<>();

		public Map<String, CompositeServiceConfig> getServices() {
			services.forEach((k, v) -> {
				if(v.getName() == null || v.getName().trim().isEmpty()) {
					v.setName(k);
				}
			});
			return services;
		}

		@lombok.Getter
		@lombok.Setter
		public static class CompositeServiceConfig {

			/** The name of the composite service. */
			private String name;
			/**
			 * The list of routes of the composite service.
			 * The first route that matches is used. Therefore the order of these routes has significance.
			 */
			private List<Route> routes = new ArrayList<>();

			public CompositeServiceConfig(String name) {
				this.name = name;
			}

			public CompositeServiceConfig() {
				// empty for constructing from YAML
			}

			@lombok.Getter
			@lombok.Setter
			public static class Route {

				/** The target service of the route. */
				private String service;
				/** The list of events/topics, which are propagated to/from the target service. */
				private List<String> events = new ArrayList<>();

			}
		}
	}

	@lombok.Getter
	@lombok.Setter
	public static class Messaging {

		/** Properties of the messaging adapter. */
		@DocumentedProperty(false)
		private Enabled receiver = new Enabled(true);
		/** Determines, if queues are deleted and recreated during startup of the application (only for development). */
		@DocumentedProperty(false)
		private boolean resetQueues = false;
		/**
		 * The list of routes for the composite messaging service.
		 * The first route that matches is used. Therefore the order of these routes has significance.
		 */
		private List<Route> routes = new ArrayList<>();
		/**
		 * Properties for messaging services.
		 * The key can be chosen arbitrarily and is used as the messaging service name, if the name property is not explicitly defined.
		 * In addition it can leveraged to split configuration across multiple profiles.
		 */
		private Map<String, MessagingServiceConfig> services = new HashMap<>();

		public Map<String, MessagingServiceConfig> getServices() {
			services.forEach((k, v) -> {
				if(v.getName() == null || v.getName().trim().isEmpty()) {
					v.setName(k);
				}
			});
			return services;
		}

		public MessagingServiceConfig getService(String name) {
			return getServices().values().stream()
					.filter(s -> Objects.equals(s.getName(), name))
					.findFirst()
					.orElse(new MessagingServiceConfig(name));
		}

		public List<MessagingServiceConfig> getServicesByKind(String kind) {
			return getServices().values().stream()
					.filter(s -> Objects.equals(s.getKind(), kind))
					.collect(Collectors.toList());
		}

		public List<MessagingServiceConfig> getServicesByBinding(String binding) {
			return getServices().values().stream()
					.filter(s -> (Objects.equals(s.getBinding(), binding) || (s.getBinding() == null && Objects.equals(s.getName(), binding))))
					.collect(Collectors.toList());
		}

		@lombok.Getter
		@lombok.Setter
		public static class MessagingServiceConfig extends Enabled {

			/** The name of the messaging service. */
			private String name;
			/**
			 * The kind of the messaging service.
			 * It usually reflects the corresponding service binding type.
			 * Possible values are: `file-based-messaging`, `enterprise-messaging`, `mqueue-sandbox`.
			 */
			private String kind;
			/**
			 * The name of the service binding used for this messaging service.
			 * In case of file-based-messaging this specifies the file-system path to the exchange file.
			 */
			private String binding;
			/** The namespace used to prefix topics from event handler registrations when creating their queue subscription. */
			private String topicNamespace;
			/** Properties for the JMS client connection. */
			private Connection connection = new Connection();
			/** Properties of the queue that is created for the messaging service. */
			private Queue queue = new Queue();
			/** Properties to control, how the outbox should be used for this service. */
			@DocumentedProperty(false)
			private Enabled outbox = new Enabled(false);

			public MessagingServiceConfig() {
				super(true);
			}

			public MessagingServiceConfig(String name) {
				super(true);
				this.name = name;
			}

			@lombok.Getter
			@lombok.Setter
			public static class Connection {

				/**
				 * Determines, if this messaging service uses its own dedicated JMS client connection.
				 * By default, JMS client connections to the same messaging broker are shared.
				 */
				private boolean dedicated = false;
				/**
				 * Properties passed to the JMS client connection.
				 * The possible keys and values depend on the messaging service implementation.
				 */
				private Map<String, String> properties = new HashMap<>();

			}

			@lombok.Getter
			@lombok.Setter
			public static class Queue {

				/**
				 * The name of the queue.
				 * The queue may already exist with some custom configuration. In that case the queue is not recreated.
				 */
				private String name;
				/** Specifies the amount of times a message from the message broker is emitted on the messaging service, before it is silently acknowledged in case of continuous errors. */
				private Integer maxFailedAttempts = 0;
				/**
				 * Properties passed to the messaging broker when creating the queue.
				 * The possible keys and values depend on the messaging service implementation.
				 */
				private Map<String, String> config = new HashMap<>();
				/**
				 * A list of additional topic, that are subscribed on the queue.
				 * By default event handler registrations should be used to trigger subscriptions.
				 * This property is intended for purposes when subscriptions can not be inferred from event handler registrations.
				 */
				private List<String> subscriptions = new ArrayList<>();

			}

		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class MultiTenancy {

		@DocumentedProperty(false)
		private String callbackUrl; // async deployment only available internally
		/** Properties of the subscription HTTP endpoints. */
		private Servlet endpoint = new Servlet("/mt/v1.0/subscriptions");
		/** Properties for the multi-tenant aware datasource. */
		private DataSource dataSource = new DataSource();
		/** Properties for the instance-manager / service-manager client. */
		private ServiceManager serviceManager = new ServiceManager();
		/** Properties for authorization. */
		private Security security = new Security();
		@DocumentedProperty(false)
		private Deployer deployer = new Deployer();
		/** Properties for the MTX sidecar client. */
		private Sidecar sidecar = new Sidecar();
		/** Properties for the URL to the application's UI endpoints. */
		private AppUi appUi = new AppUi();
		/** Properties for health check of the multi-tenant aware datasource. */
		private HealthCheck healthCheck = new HealthCheck();

		@lombok.Getter
		@lombok.Setter
		public static class DataSource {

			/**
			 * Pool to use for the multi-tenant aware datasource.
			 * Possible values are: `hikari`, `tomcat`, `atomikos`.
			 */
			private String pool = "hikari";
			/**
			 * Properties to control how the connection pools are maintained.
			 * This allows to configure that the connection pools for tenants contained in the same database are combined, instead of having a dedicated connection pool for each tenant schema.
			 */
			@DocumentedProperty(false)
			private Enabled combinePools = new Enabled(false);
			/**
			 * The list of SAP HANA database IDs.
			 * Only required when combining the connection pools of the same database.
			 */
			@DocumentedProperty(false)
			private List<String> hanaDatabaseIds = new ArrayList<>();

		}

		@lombok.Getter
		@lombok.Setter
		public static class ServiceManager {

			/** The timeout for requests in seconds. */
			private int timeout = 3600;

		}

		@Deprecated
		public ServiceManager getInstanceManager() {
			return serviceManager;
		}

		@Deprecated
		public void setInstanceManager(ServiceManager serviceManager) {
			this.serviceManager = serviceManager;
		}

		@lombok.Getter
		@lombok.Setter
		public static class Security {

			/** The scope by which the subscription endpoints are authorized. */
			private String subscriptionScope = "mtcallback";
			/** The scope by which the deployment endpoints are authorized. */
			private String deploymentScope = "mtdeployment";

		}

		@lombok.Getter
		@lombok.Setter
		public static class Deployer {

			private String url;
			private String user;
			private String password;
			private Duration asyncTimeout = Duration.ofMinutes(10);

		}

		@lombok.Getter
		@lombok.Setter
		public static class Sidecar {

			/** Properties for the sidecar CDS model and EDMX metadata caches. */
			private Cache cache = new Cache();
			/**
			 * The URL of the MTX sidecar.
			 * Setting this property in combination with a present service-manager service binding activates the MTX features.
			 */
			private String url;

			@lombok.Getter
			@lombok.Setter
			public static class Cache {

				/** The number of entries in the CDS model and EDMX metadata caches. */
				private int maxSize = 20;
				/** The lifetime of an entry in seconds after the entry's creation, the most recent replacement of its value, or its last access. */
				private int expirationTime = 600;
				/** The time in seconds after which a cached entry is refreshed. */
				private int refreshTime = 60;

			}
		}

		@lombok.Getter
		@lombok.Setter
		public static class AppUi {

			/** The URL to the application's UI, used for the 'Go to Application' link. */
			private String url;
			/**
			 * The separator for the tenant in the URL.
			 * Possible values are: `.`, `-`.
			 */
			private String tenantSeparator;

		}

		@lombok.Getter
		@lombok.Setter
		public static class HealthCheck extends Enabled {

			/** The statement that is used when executing a health check of the multi-tenant aware datasource. */
			private String healthCheckStatement = ""; // empty means chosen by mt library
			/** The time in milliseconds a health check result is cached and no further health-checks are performed. */
			private long intervalMillis = 10000;

			public HealthCheck() {
				super(true);
			}

		}
	}

	// BACKWARDS COMPATIBILITY: cds.services was renamed to cds.application.services

	@Deprecated
	public Map<String, ApplicationServiceConfig> getServices() {
		return getApplication().getServices();
	}

	@Deprecated
	public void setServices(Map<String, ApplicationServiceConfig> services) {
		getApplication().setServices(services);
	}

	@lombok.Getter
	@lombok.Setter
	public static class Application {

		/**
		 * Properties for application services.
		 * The key can be chosen arbitrarily and is used as the application service name, if the name property is not explicitly defined.
		 * In addition it can leveraged to split configuration across multiple profiles.
		 */
		private Map<String, ApplicationServiceConfig> services = new HashMap<>();

		public Map<String, ApplicationServiceConfig> getServices() {
			services.forEach((k, v) -> {
				if(v.getName() == null || v.getName().trim().isEmpty()) {
					v.setName(k);
				}
			});
			return services;
		}

		public ApplicationServiceConfig getService(String name) {
			return getServices().values().stream()
					.filter(s -> Objects.equals(s.getName(), name))
					.findFirst()
					.orElse(new ApplicationServiceConfig(name));
		}

		public List<ApplicationServiceConfig> getServicesByModel(String model) {
			return getServices().values().stream()
					.filter(s -> (Objects.equals(s.getModel(), model) || (s.getModel() == null && Objects.equals(s.getName(), model))))
					.collect(Collectors.toList());
		}

		@lombok.Getter
		@lombok.Setter
		public static class ApplicationServiceConfig extends CdsProperties.Definition {

			/**
			 * The qualified name of the CDS service, which is the model definition of this application service.
			 * It defaults to the name of the application service itself.
			 */
			private String model;
			@DocumentedProperty(false)
			private String source;
			/** Properties to configure how this service is served by protocol adapters. */
			private Serve serve = new Serve();
			/** Properties to connect messaging to this application service. */
			@DocumentedProperty(false)
			private Messaging messaging = new Messaging();

			public ApplicationServiceConfig() {
				// keep default c'tor for deserializion from property files
			}

			public ApplicationServiceConfig(String name) {
				this.name = name;
			}

			@lombok.Getter
			@lombok.Setter
			public static class Serve {

				/** Determines, if the service is ignored by protocol adapters. */
				private boolean ignore;
				/**
				 * The path this service should be served at by protocol adapters.
				 * The path is appended to the protocol adapter's base path.
				 * If a service is served by multiple protocol adapters, each adapter serves the service under this path.
				 */
				private String path;
				/**
				 * The list of protocols adapters this service should be served by.
				 * By default the service is served by all available protocol adapters.
				 * Possible values are: `odata-v4`, `odata-v2`.
				 */
				private List<String> protocols = new ArrayList<>();
				/**
				 * Properties to control more fine-grained under which endpoints this service is served.
				 * These properties override the more general properties `paths` and `protocols`.
				 */
				private List<Endpoint> endpoints = new ArrayList<>();

				@lombok.Getter
				@lombok.Setter
				public static class Endpoint {

					/**
					 * The path, this endpoint should be served at by the protocol adapter.
					 * The path is appended to the protocol adapter's base path.
					 */
					private String path;
					/**
					 * The protocol adapter that serves this endpoint.
					 * Possible values are: `odata-v4`, `odata-v2`.
					 */
					private String protocol;

				}

			}

			@lombok.Getter
			@lombok.Setter
			public static class Messaging extends CdsProperties.Definition.Messaging {

				/**
				 * The logical messaging type, used to interpret the payload of asynchronous events emitted on the service.
				 * Setting this property, implicitly enables the service to receive messages from a message broker.
				 * Possible values are: `cds` or `s4hana`.
				 */
				private String type;
				@DocumentedProperty(false)
				private String topicNamespace;

			}
		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Remote {

		/**
		 * Properties for remote services.
		 * The key can be chosen arbitrarily and is mainly intended to be able to split configuration across multiple profiles.
		 */
		private Map<String, RemoteServiceConfig> services = new HashMap<>();

		public RemoteServiceConfig getService(String name) {
			return services.values().stream()
					.filter(s -> Objects.equals(s.getName(), name))
					.findFirst()
					.orElse(new RemoteServiceConfig(name));
		}

		public List<RemoteServiceConfig> getServicesByModel(String model) {
			return services.values().stream()
					.filter(s -> (Objects.equals(s.getModel(), model) || (s.getModel() == null && Objects.equals(s.getName(), model))))
					.collect(Collectors.toList());
		}

		@lombok.Getter
		@lombok.Setter
		public static class RemoteServiceConfig {

			/** The name of the remote service. */
			private String name;
			/**
			 * The qualified name of the CDS service, which is the model definition of this remote service.
			 * It defaults to the name of the remote service itself.
			 */
			private String model;
			/** Properties to configure a remote destination for this remote service. */
			private Destination destination = new Destination();

			public RemoteServiceConfig() {
				// keep default c'tor for deserializion from property files
			}

			public RemoteServiceConfig(String name) {
				this.name = name;
			}

			@lombok.Getter
			@lombok.Setter
			public static class Destination {

				public static final String ODATA_V2_TYPE = "odata-v2";
				public static final String ODATA_V4_TYPE = "odata-v4";

				/**
				 * The protocol type of the destination.
				 * Possible values are: `odata-v4` or `odata-v2`.
				 */
				private String type = ODATA_V4_TYPE; // or ODATA_V2_TYPE
				/** The name of the destination in the destination service or Cloud SDK destination accessor. */
				private String name;
				/** A suffix for this destination, that is appended to the destination's URL. */
				private String suffix;
				/**
				 * The name of the service, that is appended to the destination's URL (after the suffix).
				 * It defaults to the qualified name of the model definition.
				 */
				private String service;

			}

		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Locales {

		/** Properties to configure how locales should be normalized. */
		private Normalization normalization = new Normalization();

		@lombok.Getter
		@lombok.Setter
		public static class Normalization {

			/** Determines, if the non-normalization include list, as described in the documentation, is applied. */
			private boolean defaults = true;
			/** The list of additional locales to add to the include list of non-normalized locales. */
			private List<String> includeList = new ArrayList<>();

			// BACKWARDS COMPATIBILITY: whiteList was renamed to includeList

			@Deprecated
			public List<String> getWhiteList() {
				return getIncludeList();
			}

			@Deprecated
			public void setWhiteList(List<String> whiteList) {
				setIncludeList(whiteList);
			}

		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Errors {

		/**
		 * Properties to configure how error messages from the framework are treated.
		 * If turned off, only framework error messages, that are explicitly localized are returned.
		 * Other errors are mapped to their plain HTTP error code representation.
		 */
		private Enabled stackMessages = new Enabled(true);
		/** Determines, if error messages are automatically extended with additional debug information (only for development). */
		private boolean extended = false;

	}

	/** Properties for drafts. */
	@lombok.Getter
	@lombok.Setter
	public static class Drafts {

		/** The maximum amount of time, since the last change, an entity instance is locked by the user who is editing its draft version. */
		private Duration cancellationTimeout = Duration.ofMinutes(15);
		/** Properties for associations from draft-enabled active entities to draft-enabled inactive entities. */
		@DocumentedProperty(false)
		private Enabled associationsToInactiveEntities = new Enabled(true);
		/** The maximum amount of time a draft is kept, before it is garbage collected. */
		private Duration deletionTimeout = Duration.ofDays(30);
		/** Properties to configure the automatic draft garbage collection. */
		private GC gc = new GC();

		@lombok.Getter
		@lombok.Setter
		public static class GC extends Enabled {

			/** The interval, in which the automatic draft garbage collection is triggered. */
			private Duration interval = Duration.ofHours(6);

			public GC() {
				super(true);
			}

		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Query {

		/** Properties for server-driven paging. */
		private Limit limit = new Limit();
		/** Properties for the implicit-sorting feature. */
		private Enabled implicitSorting = new Enabled(true);

		public static class Limit {
			/**
			 * The default page size for server-driven paging.
			 * Setting this property to 0 or -1 disables the default page size.
			 */
			private int _default = 0; // can't use lombok on this, as default is an invalid attribute name
			/**
			 * The maximum page size for server-driven paging.
			 * Setting this property to 0 or -1 disables the maximum page size.
			 */
			@lombok.Getter @lombok.Setter private int max = 1000;

			public int getDefault() {
				return _default;
			}

			public void setDefault(int _default) {
				this._default = _default;
			}
		}

	}

	/*
	 * COMMON CONFIG POJOS
	 */
	@lombok.Setter
	public static class Enabled {

		/** Determines, if it is enabled. */
		private Boolean enabled; // can't use lombok on this, as it would generate getEnabled() instead of isEnabled()

		public Enabled(Boolean byDefault) {
			this.enabled = byDefault;
		}

		public Boolean isEnabled() {
			return enabled;
		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Servlet extends Enabled {

		/** The base-path of the adapter endpoint. */
		private String path;

		public Servlet(String defaultPath) {
			super(true);
			this.path = defaultPath;
		}

	}

	@lombok.Getter
	@lombok.Setter
	public static class Definition {

		/** The name of the CDS definition. */
		protected String name;
		/** Properties to connect messaging to this CDS definition. */
		@DocumentedProperty(false)
		protected Messaging messaging = new Messaging();
		@DocumentedProperty(false)
		protected Map<String, Definition> definitions = new HashMap<>();

		public Definition getDefinition(String name) {
			return definitions.values().stream().filter(d -> Objects.equals(d.getName(), name)).findFirst().orElse(new Definition());
		}

		@lombok.Getter
		@lombok.Setter
		public static class Messaging extends Enabled {

			@DocumentedProperty(false)
			private List<String> services = new ArrayList<>();
			@DocumentedProperty(false)
			private List<String> topics = new ArrayList<>();

			public Messaging() {
				super(null); // by default look into the model
			}

		}
	}
}
