/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.localization;

import java.text.MessageFormat;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.helpers.MessageFormatter;

import com.sap.cds.feature.config.Properties;

public class DefaultLocalizedMessageProvider implements LocalizedMessageProvider {

	private final static Logger logger = LoggerFactory.getLogger(DefaultLocalizedMessageProvider.class);

	private final Map<LocaleCode, MessageFormat> cachedMessageFormats = new ConcurrentHashMap<>();

	private static final MessageFormat NotFoundMessageFormat = new MessageFormat("<not found>");

	final static Locale SAPTRC_LOCALE = Locale.forLanguageTag("en-US-x-saptrc");
	final static Locale SAPTRC_BUNDLE = Locale.forLanguageTag("en-US-saptrc");

	final static Locale SAPPSD_LOCALE = Locale.forLanguageTag("en-US-x-sappsd");
	final static Locale SAPPSD_BUNDLE = Locale.forLanguageTag("en-US-sappsd");

	/**
	 * Determines the locale used for resource bundle lookups
	 * @param locale the locale
	 * @return the locale to beused for bundle lookup
	 */
	public static Locale getLocaleForBundle(Locale locale) {
		if(SAPTRC_LOCALE.equals(locale)) {
			return SAPTRC_BUNDLE;
		} else if(SAPPSD_LOCALE.equals(locale)) {
			return SAPPSD_BUNDLE;
		}
		return locale;
	}

	@Override
	public String getMessage(String code, Object[] args, Locale locale) {

		if (code == null) {
			return null;

		} else if (locale == null) {
			locale = Locale.getDefault();
		}

		try {
			MessageFormat messageFormat = findMessageFormat(code, getLocaleForBundle(locale));
			if (messageFormat != null) {
				synchronized(messageFormat) { // MessageFormat is not thread-safe
					Object[] theArgs = (args != null && args.length > 0) ? args : new Object[0];
					return messageFormat.format(theArgs);
				}
			} else {
				return MessageFormatter.arrayFormat(code, args).getMessage();
			}

		} catch (Exception e) {  // NOSONAR
			// be tolerant - we don't want to break the request due to wrong message resolution
			logger.warn("Invalid message format '{}' or incompatible arguments {}", code, toStringSafely(args), e);
			return code;
		}

	}

	private static String toStringSafely(Object[] args) {
		if (args != null) {
			StringBuffer sb = new StringBuffer();
			for (Object arg : args) {
				sb.append('<');
				try {
					sb.append(arg.toString());
				} catch(Exception e) { // NOSONAR
					sb.append(e.getMessage());
				}
				sb.append('>');
			}
			return sb.toString();
		}
		return "<null>";
	}

	@Override
	public boolean isActiveFeature() {
		return true;
	}

	@Override
	public String getFeatureName() {
		return "Default Localized Message Provider";
	}

	protected MessageFormat findMessageFormat(String code, Locale locale) {

		LocaleCode localeCode = new LocaleCode(code, locale);
		MessageFormat cachedMessageFormat = cachedMessageFormats.get(localeCode);
		if (cachedMessageFormat == null) {
			// try to resolve once
			cachedMessageFormat = resolveMessageFormat(code, locale);
			if (cachedMessageFormat == null) {
				cachedMessageFormats.put(localeCode, NotFoundMessageFormat);
			} else {
				cachedMessageFormats.put(localeCode, cachedMessageFormat);
			}
		}
		else if (cachedMessageFormat == NotFoundMessageFormat) {
			// will never resolve
			return null;
		}

		return cachedMessageFormat;
	}

	protected MessageFormat resolveMessageFormat(String code, Locale locale) {

		List<String> basenames = Properties.getPlain().getMessages().getBasenames();
		for (String basename : basenames) {
			try {
				ResourceBundle bundle = ResourceBundle.getBundle(basename, locale);
				if (bundle.containsKey(code)) {
					try {
						String msg = bundle.getString(code);
						return new MessageFormat(msg, locale);

					} catch (MissingResourceException e) {  // NOSONAR
						// key still seems to be missing - which is not expected
						logger.warn("Failed to lookup '{}' from bundle '{}'", code, bundle.getBaseBundleName(), e);
					}
				}
			}
			catch (MissingResourceException ex) {
				logger.warn("ResourceBundle '{}' not found: {} ", basename, ex.getMessage());
			}
		}

		return null;
	}

	private static class LocaleCode {
		String code;
		Locale locale;

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((code == null) ? 0 : code.hashCode());
			result = prime * result + ((locale == null) ? 0 : locale.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			LocaleCode other = (LocaleCode) obj;
			if (code == null) {
				if (other.code != null)
					return false;
			} else if (!code.equals(other.code))
				return false;
			if (locale == null) {
				if (other.locale != null)
					return false;
			} else if (!locale.equals(other.locale))
				return false;
			return true;
		}

		LocaleCode(String code, Locale locale) {
			this.code = code;
			this.locale = locale;
		}
	}
}
