/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.platform.local;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Scanner;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.feature.config.Properties;
import com.sap.cds.feature.platform.ApplicationProperties;
import com.sap.cds.feature.platform.PlatformEnvironment;
import com.sap.cds.feature.platform.ServiceBinding;
import com.sap.cds.feature.platform.util.VcapApplicationParser;
import com.sap.cds.feature.platform.util.VcapServicesParser;

public class LocalPlatformEnvironment implements PlatformEnvironment {

	private final static String defaultEnv = "default-env.json";
	private final static Logger logger = LoggerFactory.getLogger(LocalPlatformEnvironment.class);

	private ApplicationProperties properties;
	private List<ServiceBinding> bindings;

	@Override
	public ApplicationProperties getApplicationProperties() {
		if(properties == null) {
			properties = VcapApplicationParser.getApplicationProperties(getDefaultEnv());
		}
		return properties;
	}

	@Override
	public Stream<ServiceBinding> getServiceBindings() {
		if(bindings == null) {
			bindings = VcapServicesParser.getServiceBindings(getDefaultEnv());
		}
		return bindings.stream();
	}

	private String getDefaultEnv() {
		// first look inside the resources folder
		InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream(defaultEnv);

		if (stream == null) {
			// look at the configured path
			File defaultEnvFile = null;
			String defaultEnvPath = Properties.getCds().getEnvironment().getLocal().getDefaultEnvPath();
			if (defaultEnvPath != null) {
				// check, if the path exists and is a file or directory
				defaultEnvFile = new File(defaultEnvPath);
				if (defaultEnvFile.isDirectory()) {
					// configured as directory, use default file name
					defaultEnvFile = Paths.get(defaultEnvPath).resolve(defaultEnv).toFile();
				}
				if (defaultEnvFile.exists()) {
					try {
						stream = new FileInputStream(defaultEnvFile);
					} catch (FileNotFoundException e) { // NOSONAR
						// ignore
					}
				} else {
					// a path property was configured, but no file was found
					logger.warn("No default env file found in configured path '{}'", defaultEnvPath);
				}
			}

			// now try looking inside the working directory
			Path workingDirectory = Paths.get(System.getProperty("user.dir"));

			// try within working directory
			if(stream == null) {
				defaultEnvFile = workingDirectory.resolve(defaultEnv).toFile();
				if(defaultEnvFile.exists()) {
					try {
						stream = new FileInputStream(defaultEnvFile);
					} catch (FileNotFoundException e) { // NOSONAR
						// ignore
					}
				}
			}

			// try within parent of working directory
			if(stream == null) {
				defaultEnvFile = workingDirectory.getParent().resolve(defaultEnv).toFile();
				if (defaultEnvFile.exists()) {
					try {
						stream = new FileInputStream(defaultEnvFile);
					} catch (FileNotFoundException e) { // NOSONAR
						// ignore
					}
				}
			}

			if(stream != null && defaultEnvFile != null) { // NOSONAR
				logger.info("Loaded {} from directory '{}'", defaultEnvFile.getName(), defaultEnvFile.getParent());
			}
		} else {
			logger.info("Loaded {} from classpath resources", defaultEnv);
		}

		String defaultEnvString = null;
		if(stream != null) {
			try(Scanner scanner = new Scanner(stream, StandardCharsets.UTF_8.name())) {
				defaultEnvString = scanner.useDelimiter("\\A").next();
			}
		}

		return defaultEnvString;
	}

	@Override
	public boolean isActiveFeature() {
		return true;
	}

	@Override
	public String getFeatureName() {
		return "Local Platform Environment (default-env.json)";
	}

}
