/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.platform.util;

import java.io.IOException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.feature.platform.ApplicationProperties;

/**
 * Parser for CloudFoundry's VCAP_APPLICATION
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class VcapApplicationParser implements ApplicationProperties {

	public final static String VCAP_APPLICATION = "VCAP_APPLICATION";

	private static final Logger logger = LoggerFactory.getLogger(VcapApplicationParser.class);
	private static final ObjectMapper mapper = new ObjectMapper();

	public static ApplicationProperties getApplicationProperties(String vcapApplication) {
		if (vcapApplication != null && vcapApplication.trim().length() > 0) {
			try {
				JsonNode vcapNodes = mapper.readTree(vcapApplication);

				// if the VCAP_APPLICATION key is part of the JSON, use its value
				if(vcapNodes.has(VCAP_APPLICATION)) {
					vcapNodes = vcapNodes.get(VCAP_APPLICATION);
				}

				return mapper.treeToValue(vcapNodes, VcapApplicationParser.class);
			} catch (IOException e) {
				logger.warn("Could not parse VCAP_APPLICATION.", e);
			}
		}

		return new VcapApplicationParser();
	}

	@JsonProperty("application_id")
	private String applicationId = "local";

	@JsonProperty("application_name")
	private String applicationName = "local-" + System.getProperty("user.name");

	@Override
	public String getApplicationId() {
		return applicationId;
	}

	@Override
	public String getApplicationName() {
		return applicationName;
	}
}
