/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.platform.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.feature.platform.ServiceBinding;

/**
 * Parser for Open Service Broker API service bindings
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public class VcapServicesParser implements ServiceBinding  {

	public final static String VCAP_SERVICES = "VCAP_SERVICES";

	private static final Logger logger = LoggerFactory.getLogger(VcapServicesParser.class);
	private static final ObjectMapper mapper = new ObjectMapper();

	public static List<ServiceBinding> getServiceBindings(String vcapServices) {
		List<ServiceBinding> bindings = new ArrayList<>();

		if (vcapServices != null && vcapServices.trim().length() > 0) {
			try {
				JsonNode vcapNodes = mapper.readTree(vcapServices);

				// if the VCAP_SERVICES key is part of the JSON, use its value
				if(vcapNodes.has(VCAP_SERVICES)) {
					vcapNodes = vcapNodes.get(VCAP_SERVICES);
				}

				// for each service a field exists
				vcapNodes.fields().forEachRemaining(serviceField -> {

					// for each binding an entry in the array exists
					Iterator<JsonNode> bindingNodes = serviceField.getValue().elements();
					int i = 0;
					while(bindingNodes.hasNext()) {
						try {
							VcapServicesParser binding = mapper.treeToValue(bindingNodes.next(), VcapServicesParser.class);
							binding.setService(serviceField.getKey());
							bindings.add(binding);
						} catch (IOException e) {
							logger.warn("Could not parse service binding of service '{}' at index ''.", serviceField.getKey(), i, e);
						}
						++i;
					}
				});
			} catch (IOException e) {
				logger.warn("Could not parse " + VCAP_SERVICES, e);
			}
		}

		return bindings;
	}

	private String service;

	@JsonProperty("name")
	private String name;

	@JsonProperty("plan")
	private String servicePlan;

	@JsonProperty("tags")
	private List<String> tags;

	@JsonProperty("credentials")
	private Map<String, Object> credentials;

	@Override
	public String getName() {
		return name;
	}

	@Override
	public String getService() {
		return service;
	}

	private void setService(String service) {
		this.service = service;
	}

	@Override
	public String getServicePlan() {
		return servicePlan;
	}

	@Override
	public List<String> getTags() {
		return tags; // NOSONAR
	}

	@Override
	public Map<String, Object> getCredentials() {
		return credentials;
	}

}
