/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.auditlog.v2;

import java.util.Collections;
import java.util.Map;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.mt.DependenciesEventContext;
import com.sap.cds.services.mt.DeploymentService;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;

/**
 * AuditLog v2 related implementation of the multi tenancy subscription handler which is responsible for providing the
 * dependencies to the bound AuditLog v2 service.
 */
@ServiceName(DeploymentService.DEFAULT_NAME)
class AuditLogV2MtHandler implements EventHandler {

	private static final Logger logger = LoggerFactory.getLogger(AuditLogV2MtHandler.class);

	private final ServiceBinding binding;

	AuditLogV2MtHandler(ServiceBinding binding) {
		this.binding = Objects.requireNonNull(binding, "binding must not be null");
	}

	@After
	@SuppressWarnings("unchecked")
	public void addAuditLogV2Dependency(DependenciesEventContext context) {
		Map<String, String> uaa = (Map<String, String>) this.binding.getCredentials().get("uaa");
		String xsappname = uaa != null ? uaa.get("xsappname") : null;

		if (xsappname != null) {
			logger.debug("Providing AuditLog v2 service binding dependency '{}' to xsappname '{}'",
					this.binding.getName().orElse(""), xsappname);
			context.getResult().add(Collections.singletonMap("xsappname", xsappname));
		} else {
			logger.warn("xsappname missing in service binding: '{}'", this.binding.getName().orElse(""));
		}
	}

}
