/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.auditlog.v2;

import java.net.URI;
import java.util.Map;

import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import com.sap.cds.repackaged.audit.client.impl.Utils;

class AuditLogV2Utils {
    /**
     * 'premium' plan is not yet defined in {@link Utils}.
     */
    static final String PREMIUM_PLAN = "premium";

    static String getServiceUrl(Map<String, Object> credentials, boolean oAuth2) {
        String url = (String) credentials.get("url");

        URI uri = URI.create(url).normalize();
        String auditServicePath = Utils.AUDIT_SERVICE_URL_PREFIX + (oAuth2 ? "oauth2/" : "") + "v2/";
        if (!uri.getPath().endsWith(auditServicePath)) {
            uri = uri.resolve(auditServicePath);
        }
        return uri.toASCIIString();
    }

    static boolean isOAuth2BasedServicePlan(ServiceBinding binding) {
        String servicePlan = binding.getServicePlan().orElse(null);

        return (Utils.OAUTH2_PLAN.equals(servicePlan) || PREMIUM_PLAN.equals(servicePlan));
    }

    static boolean isStandardPlan(ServiceBinding binding) {
        String servicePlan = binding.getServicePlan().orElse(null);

        return Utils.STANDARD_PLAN.equals(servicePlan);
    }

    static void validateBinding(ServiceBinding binding) {
        Map<String, Object> cred = binding.getCredentials();
        if (cred == null || cred.isEmpty()) {
            throw new ErrorStatusException(CdsErrorStatuses.AUDITLOG_SERVICE_INVALID_CONFIG, "credentials");
        }

        if (!cred.containsKey("url") || StringUtils.isEmpty((String) cred.get("url"))) {
            throw new ErrorStatusException(CdsErrorStatuses.AUDITLOG_SERVICE_INVALID_CONFIG, "credentials.url");
        }

        if (isOAuth2BasedServicePlan(binding)) {
            @SuppressWarnings("unchecked")
            Map<String, Object> uaa = (Map<String, Object>) cred.get("uaa");
            if (uaa == null || uaa.isEmpty()) {
                throw new ErrorStatusException(CdsErrorStatuses.AUDITLOG_SERVICE_INVALID_CONFIG, "credentials.uaa");
            }
        }
    }
}
