package com.sap.cds.repackaged.audit.api;

import java.time.Instant;

import com.sap.cds.repackaged.audit.api.exception.AuditLogNotAvailableException;
import com.sap.cds.repackaged.audit.api.exception.AuditLogWriteException;

public interface AuditLogMessage {
	/**
	 * Persists the already created and populated with relevant data audit message.
	 *
	 * @throws AuditLogWriteException        In case a problem occurs while persisting the audit message.
	 * @throws AuditLogNotAvailableException If the audit log server is not available.
	 */
	void log() throws AuditLogNotAvailableException, AuditLogWriteException;

	/**
	 * Sets the user that triggered the audit event. <br>
	 * <ul><li><b>[NOTE]</b> When using the <b>standard</b> plan of the auditlog service you should specify the user yourself</li>
	 * <li><b>[NOTE]</b> When using the <b>oauth2</b> plan of the auditlog service you should write <b><code>setUser("$USER")</code></b> and the user will be set automatically on server side</li></ul>
	 *
	 * @param user <ul><li>The value should be the user name when using the <b>standard</b> service plan</li>
	 * <li> The value should be "$USER" when using the <b>oauth2</b> service plan</li></ul>
	 */
	void setUser(String user);

	/**
	 * Sets the identity provider name for the user that triggered the audit event.
	 * <b>[NOTE]</b> When using standard plan of the auditlog service you should specify the identity provider (IdP) name yourself. <br>
	 * When using <b>oauth2</b> plan of the auditlog service you should write <b>setIdentityProvider("$IDP")</b> and the IdP name will be set automatically on server side.
	 *
	 * @param identityProvider The identity provider where the user is defined.
	 */
	void setIdentityProvider(String identityProvider);

	/**
	 * Sets the tenant that triggered the audit event. <br>
	 * <ul><li><b>[NOTE]</b> When using the <b>standard</b> plan of the auditlog service you should specify the tenant yourself - <b><code>setTenant("tenant-guid-here")</code></b>. </li>
	 * <li><b>[NOTE]</b> When using the <b>oauth2</b> plan of the auditlog service this method should be used to set the tenant of the message to the <b>PROVIDER</b> tenant
	 * You should write <b><code>setTenant("$PROVIDER")</code></b> if you want to log the message for the provider and the tenant will be set automatically on server side. 
	 * If you want to log the message for the subscriber, please use the {@link #setTenant(String tenant, String subscriberTokenIssuer) setTenant(tenant, subscriberTokenIssuer)} method</li></ul>
	 *
	 * @param tenant <ul><li>The value should be the tenant guid when using the <b>standard</b> service plan</li>
	 * <li> The value should be "$PROVIDER" when using the <b>oauth2</b> service plan</li></ul>
	 */
	void setTenant(String tenant);

	/**
	 * This method should be used to set the tenant of the message to the <b>SUBSCRIBER</b> tenant <b>ONLY</b> when using the <b>oauth2</b> service plan of the auditlog service. <br>
	 * <b>[NOTE]</b> You should write <b><code>setTenant("$SUBSCRIBER", subscriberTokenIssuer)</code></b> if you want to log the message for the subscriber and the tenant will be set automatically on server side
	 *
	 * @param tenant The value should be "$SUBSCRIBER".
	 * @param subscriberTokenIssuer The token issuer for the subscriber tenant. E.g.: https://subdomain.authentication.stagingaws.hanavlab.ondemand.com 
	 * <ul><li>The subscriber token issuer should be the token issuer of a real subscribed tenant</li>
	 * <li>The subscriber token issuer should not contain "/oauth/token" as it is automatically added by the XSUAA</li></ul>
	 */
	@Deprecated
	void setTenant(String tenant, String subscriberTokenIssuer);

	/**
	 * Set the tenant of the message to the <b>SUBSCRIBER</b> tenant when using the <b>oauth2</b> service plan and automatically builds the subscriber token issuer based on the provided subaccount subdomain.
	 * Sets the tenant field to "$SUBSCRIBER"
	 * Sets subscriber token issuer depending on the credential type used. E.g:
	 * <ul><li>"subdomain.authentication.stagingaws.hanavlab.ondemand.com" for client credentials</li>
	 * <li>"subdomain.authentication.cert.stagingaws.hanavlab.ondemand.com" for X509 mTLS</li></ul>
	 * <b>[IMPORTANT]</b> This method replaces <b><code>setTenant(tenant, subscriberTokenIssuer)</code></b> and should be used ONLY when writing messages for the SUBSCRIBER tenant.
	 * @param subdomain The subdomain of the subscriber tenant. The subscriber subdomain should be the subdomain of a real subscribed tenant
	 */

	void setTenantBySubscriberSubdomain(String subdomain);

	/**
	 * Set custom specific data.
	 *
	 * @param key Key of custom field
	 * @param value Custom string or json. The value should be represented in a Jackson serializable/deserializable manner.
	 */
	void addCustomDetails(String key, Object value);

	/**
	 * @deprecated Use com.sap.cds.repackaged.audit.api.AuditLogMessage.setEventTime instead
	 * @param time
	 */
	@Deprecated
	void setTime(Instant time);

	/**
	 * @deprecated The UUID is always automatically generated by the service and can't be changed.
	 * @param uuid
	 */
	@Deprecated
	void setUuid(String uuid);

	/**
	 * Set the time of the event which is being logged.
	 *
	 * @param time The time of the event
	 */
	void setEventTime(Instant time);
}
