package com.sap.cds.repackaged.audit.api.v2;

import com.sap.cds.repackaged.audit.api.AuditLogMessage;

/**
 * Data access interface for logging successful and unsuccessful access to any sensitive personal data. Logging shall be done where data is disclosed to a user or a third party.
 * 
 * Examples of such personal data include: information on racial or ethnic origin, political opinions, religious or philosophical beliefs, trade-union membership, health or sex
 * life, bank account data and any other data that can identify certain individual or entity.
 */
public interface DataAccessAuditMessage extends AuditLogMessage {

	/**
	 * Sets the data access channel type. For example - RFC, web service, IDOC, file based interfaces, user interface, spool, printing, etc.
	 * 
	 * @param key The type of data access channel.
	 */
	void setChannel(String key);

	/**
	 * Sets the audited object where the accessed personal data is contained.
	 * 
	 * @param object The object where the accessed personal data that is being audit logged is contained.
	 */
	void setObject(AuditedObject object);

	/**
	 * Sets the audited data subject which is the owner of the accessed personal data audit logged with this event.
	 * 
	 * @param dataSubject The data subject whom the accessed personal data that is being audit logged belongs to.
	 */
	void setDataSubject(AuditedDataSubject dataSubject);

	/**
	 * Sets the heading name for the attribute that has been read, either successfully or not. Add the audited data subject in the list of data subjects whose personal data
	 * reflected in the object field have been accessed by the user.
	 * 
	 * @param dataSubject One of the multiple data subjects whose personal data is accessed that to be added in the list.
	 */
	void addDataSubject(AuditedDataSubject dataSubject);

	/**
	 * Sets the heading name for the attribute that has been read
	 * 
	 * @param name The attribute name.
	 */
	void addAttribute(String name);

	/**
	 * Sets the heading name for the attribute that has been read, either successfully or not
	 * 
	 * @param name The attribute name.
	 * @param isSuccessful Whether or not the access to the attribute was successful.
	 */
	void addAttribute(String name, boolean isSuccessful);

	/**
	 * Sets the attachment name and id in case the event is triggered by the download or display of some attachments or files
	 * 
	 * @param id Attachment id.
	 * @param name Attachment name.
	 */
	void addAttachment(String id, String name);

}
