package com.sap.cds.repackaged.audit.client.impl;

import com.sap.cds.repackaged.audit.api.exception.AuditLogNotAvailableException;
import com.sap.cds.repackaged.audit.api.exception.AuditLogWriteException;

/***
 * Provides a template for communicator tasked with easing request exchange between the Audit Log
 * Service and the client.
 *
 */
public interface Communicator {

    /**
     * 
     * Send an AuditLog message to a given endpoint.
     * 
     * 
     * 
     * @param message
     *            HTTP request body in JSON data format following the predefined schema. Contains
     *            the message to be sent and logged by the Audit Log Service.
     * @param endpoint
     *            The endpoint URL to which a request is to be made in order to log the message in
     *            the correct subcategory. Parameter is not the full URL but the service URL formed
     *            as follows: For standard plan /audit-log/v2/ For oauth2 plan /audit-log/oauth2/v2/
     * @param subscriberTokenIssuer
     *            Issuer of the JWT token for a multi-tenant application. The subscriber token
     *            issuer should be the token issuer of a real subscribed tenant. There should be a
     *            SaaS dependency to the Audit Log service in case the oauth2 service plan is used
     *            in a multi-tenant application/service. Parameter is not needed when using the
     *            standard plan and can be passed as <i>null</i>. If a custom communicator is
     *            implemented using the oauth2 plan, the value can be <i>null</> if the
     *            authentication in front of the audit log server is a responsibility of a third
     *            party.
     * @return
     * @throws AuditLogNotAvailableException
     *             Exception thrown when the audit log server is not available.
     * @throws AuditLogWriteException
     *             Thrown when the audit log message cannot be delivered to the destination storage.
     * @throws UnsupportedOperationException
     *             Thrown only by implementations that do not handle non-null
     *             <b>subscriberTokenIssuer</b>! Thrown when the value passed for
     *             <b>subscriberTokenIssuer</b> is not <i>null</i>. A <i>non-null</i> can be passed
     *             when the implementation handles value of <b>subscriberTokenIssuer</b>
     */
    public String send(String message, String endpoint, String subscriberTokenIssuer)
            throws AuditLogNotAvailableException, AuditLogWriteException, UnsupportedOperationException;
    /**
     * Get the clientId used by the communicator.
     *
     */
    public String getClientId();

    /**
     * Get the service URL base used by the communicator. E.g: http://localhost:8080/audit-log/v2/
     * 
     */
    public String getServiceUrl();

    /**
     * Get the service plan, standard or oauth2, used by the communicator.
     * 
     */
    public String getServicePlan();

    /**
     * Get uaaDomain if given given audit log service key is not of type x509. If given key is of
     * type x509 the method will build the required domain for x509 type key.
     */
    public String getUaaDomain();

    /**
     * Checks if a given audit log service key is of type x509
     * 
     * @throws AuditLogWriteException
     *             Thrown when the given key is either null or invalid.
     * 
     */
    public boolean isX509CredentialType() throws AuditLogWriteException;
}
