package com.sap.cds.repackaged.audit.client.impl;

import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.CONNECTION_KEEP_ALIVE_TIME;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.CONNECTION_KEEP_ALIVE_TIME_VALUE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.CONNECTION_REQUEST_TIMEOUT;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.CONNECTION_REQUEST_TIMEOUT_VALUE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.CONNECT_TIMEOUT;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.CONNECT_TIMEOUT_VALUE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.DELAY_RETRY_ON_EXCEPTION;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.DELAY_RETRY_ON_EXCEPTION_VALUE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.DELAY_RETRY_ON_HTTP_ERROR_RESPONSE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.DELAY_RETRY_ON_HTTP_ERROR_RESPONSE_VALUE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.HTTP_POOL_MAX_CONN;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.HTTP_POOL_MAX_CONN_PER_ROUTE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.HTTP_POOL_MAX_CONN_PER_ROUTE_VALUE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.HTTP_POOL_MAX_CONN_VALUE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.MAX_RETRY_COUNT_ON_EXCEPTION;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.MAX_RETRY_COUNT_ON_EXCEPTION_VALUE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.MAX_RETRY_COUNT_ON_HTTP_ERROR_RESPONSE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.MAX_RETRY_COUNT_ON_HTTP_ERROR_RESPONSE_VALUE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.SOCKET_TIMEOUT;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.SOCKET_TIMEOUT_VALUE;
import static com.sap.cds.repackaged.audit.client.impl.ConnectionConfigLoaderConstants.XS_AUDIT_LOG_CLIENT_CONFIG_ENV;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;

public class ConnectionConfigLoader {

    private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionConfigLoader.class);

    private Map<String, Integer> connectionPropertiesMap;

    public ConnectionConfigLoader(String env) {
        this.connectionPropertiesMap = createMapWithDefaultValues();
        readConfigurationsFromEnv(env);
    }

    private Map<String, Integer> createMapWithDefaultValues() {
        final Map<String, Integer> map = new HashMap<>();
        map.put(MAX_RETRY_COUNT_ON_HTTP_ERROR_RESPONSE, MAX_RETRY_COUNT_ON_HTTP_ERROR_RESPONSE_VALUE);
        map.put(MAX_RETRY_COUNT_ON_EXCEPTION, MAX_RETRY_COUNT_ON_EXCEPTION_VALUE);
        map.put(DELAY_RETRY_ON_HTTP_ERROR_RESPONSE, DELAY_RETRY_ON_HTTP_ERROR_RESPONSE_VALUE);
        map.put(DELAY_RETRY_ON_EXCEPTION, DELAY_RETRY_ON_EXCEPTION_VALUE);
        map.put(CONNECTION_KEEP_ALIVE_TIME, CONNECTION_KEEP_ALIVE_TIME_VALUE);
        map.put(CONNECTION_REQUEST_TIMEOUT, CONNECTION_REQUEST_TIMEOUT_VALUE);
        map.put(CONNECT_TIMEOUT, CONNECT_TIMEOUT_VALUE);
        map.put(SOCKET_TIMEOUT, SOCKET_TIMEOUT_VALUE);
        map.put(HTTP_POOL_MAX_CONN, HTTP_POOL_MAX_CONN_VALUE);
        map.put(HTTP_POOL_MAX_CONN_PER_ROUTE, HTTP_POOL_MAX_CONN_PER_ROUTE_VALUE);
        return map;
    }

    private void readConfigurationsFromEnv(String env) {
        final String auditClientConfig = (env == null ? System.getenv(XS_AUDIT_LOG_CLIENT_CONFIG_ENV) : env);

        if (auditClientConfig != null && !auditClientConfig.isEmpty()) {
            final JsonFactory jsonFactory = new JsonFactory();
            try (final JsonParser parser = jsonFactory.createParser(auditClientConfig)) {
                while (parser.nextToken() != null) {
                    if (!isTokenScalarValue(parser)) {
                        continue;
                    }

                    final String connectionProperty = String.valueOf(parser.getCurrentName());
                    if (connectionPropertiesMap.containsKey(connectionProperty)) {
                        connectionPropertiesMap.put(connectionProperty, parser.getValueAsInt());
                    }
                }
            } catch (IOException e) {
                LOGGER.error(XS_AUDIT_LOG_CLIENT_CONFIG_ENV + " environment var could not be parsed!", e);
            }
        }
    }

    private boolean isTokenScalarValue(final JsonParser parser) {
        return parser.getCurrentToken()
                .isScalarValue();
    }

    public int getMaxRetryCountOnHttpErrorResponse() {
        return connectionPropertiesMap.get(MAX_RETRY_COUNT_ON_HTTP_ERROR_RESPONSE);
    }

    public int getMaxRetryCountOnException() {
        return connectionPropertiesMap.get(MAX_RETRY_COUNT_ON_EXCEPTION);
    }

    public int getDelayRetryOnHttpErrorResponse() {
        return connectionPropertiesMap.get(DELAY_RETRY_ON_HTTP_ERROR_RESPONSE);
    }

    public int getDelayRetryOnException() {
        return connectionPropertiesMap.get(DELAY_RETRY_ON_EXCEPTION);
    }

    public int getConnectionKeepAliveTime() {
        return connectionPropertiesMap.get(CONNECTION_KEEP_ALIVE_TIME);
    }

    public int getConnectionRequestTimeout() {
        return connectionPropertiesMap.get(CONNECTION_REQUEST_TIMEOUT);
    }

    public int getConnectTimeout() {
        return connectionPropertiesMap.get(CONNECT_TIMEOUT);
    }

    public int getSocketTimeout() {
        return connectionPropertiesMap.get(SOCKET_TIMEOUT);
    }

    public int getHttpPoolMaxConn() {
        return connectionPropertiesMap.get(HTTP_POOL_MAX_CONN);
    }

    public int getHttpPoolMaxConnPerRoute() {
        return connectionPropertiesMap.get(HTTP_POOL_MAX_CONN_PER_ROUTE);
    }

}
