package com.sap.cds.repackaged.audit.client.impl;

import static com.sap.cds.repackaged.audit.client.impl.Utils.APPLICATION_JSON;
import static org.apache.http.HttpHeaders.CONTENT_TYPE;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.List;

import javax.net.ssl.SSLContext;

import org.apache.http.Header;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicHeader;

import com.sap.cloud.security.mtls.SSLContextFactory;
import com.sap.cds.repackaged.audit.api.exception.ClientInitializationException;

public class HttpClientFactory {
    private static final String HTTPS_SCHEME = "https";
    private static final String PARSING_CREDENTIALS_ERROR = "Problem occurred while trying to parse XSUAA credentials";
    private static final String SECURITY_ERROR = "Security error has occured";

    private SSLContextFactory sslContextFactory;
    private OAuthCredentials oauthCredentials;
    private ConnectionConfigLoader configLoader;
    private List<Header> httpHeaders;

    public HttpClientFactory(SSLContextFactory sslContextFactory, OAuthCredentials oauthCredentials, ConnectionConfigLoader configLoader) {
        this.sslContextFactory = sslContextFactory;
        this.oauthCredentials = oauthCredentials;
        this.configLoader = configLoader;
        this.httpHeaders = getDefaultHeaders();
    }

    public HttpClient createHttpClient(PoolingHttpClientConnectionManager connectionManager) {
        final HttpClientBuilder customHttpClientBuilder = createHttpClientBuilder(connectionManager);
        return customHttpClientBuilder.build();
    }

    public HttpClient createHttpClientWithSSLContext(PoolingHttpClientConnectionManager connectionManager) {
        final HttpClientBuilder customHttpClientBuilder = createHttpClientBuilder(connectionManager);
        try {
            enhanceClientWithSSLContext(customHttpClientBuilder, oauthCredentials);
        } catch (IOException e) {
            throw new ClientInitializationException(PARSING_CREDENTIALS_ERROR, e);
        } catch (GeneralSecurityException e) {
            throw new ClientInitializationException(SECURITY_ERROR, e);
        }

        return customHttpClientBuilder.build();
    }

    private HttpClientBuilder createHttpClientBuilder(PoolingHttpClientConnectionManager connectionManager) {
        return HttpClients.custom()
                .setConnectionManager(connectionManager)
                .setDefaultRequestConfig(getRequestConfig())
                .setKeepAliveStrategy((resp, ctx) -> configLoader.getConnectionKeepAliveTime())
                .setDefaultHeaders(httpHeaders);
    }

    private void enhanceClientWithSSLContext(HttpClientBuilder customHttpClientBuilder, OAuthCredentials oauthCredentials)
            throws GeneralSecurityException, IOException {
        PoolingHttpClientConnectionManager connectionManager;
        if (oauthCredentials != null && oauthCredentials.getCertificate() != null
                && oauthCredentials.getKey() != null) {

            final SSLContext sslContext = sslContextFactory.create(oauthCredentials.getCertificate(), oauthCredentials.getKey());
            customHttpClientBuilder.setSSLContext(sslContext);
            final SSLConnectionSocketFactory sslConnectionFactory =
                new SSLConnectionSocketFactory(sslContext);
            final Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory> create()
                    .register(HTTPS_SCHEME, sslConnectionFactory)
                    .build();
            connectionManager = new PoolingHttpClientConnectionManager(registry);
            connectionManager.setMaxTotal(configLoader.getHttpPoolMaxConn());
            connectionManager.setDefaultMaxPerRoute(configLoader.getHttpPoolMaxConnPerRoute());
            customHttpClientBuilder.setConnectionManager(connectionManager);
        }
    }

    private RequestConfig getRequestConfig() {
        return RequestConfig.custom()
                .setConnectionRequestTimeout(configLoader.getConnectionRequestTimeout())
                .setConnectTimeout(configLoader.getConnectTimeout())
                .setSocketTimeout(configLoader.getSocketTimeout())
                .build();
    }

    private List<Header> getDefaultHeaders() {
        final List<Header> headers = new ArrayList<>();
        final Header contentHeader = new BasicHeader(CONTENT_TYPE, APPLICATION_JSON);
        headers.add(contentHeader);

        return headers;
    }

}
