package com.sap.cds.repackaged.audit.client.impl;

import static com.sap.cds.repackaged.audit.client.impl.Utils.INVALID_SUBSCRIBER_TOKEN_ISSUER_ERROR;

import java.net.URI;
import java.net.URISyntaxException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.security.config.CredentialType;
import com.sap.cloud.security.config.OAuth2ServiceConfiguration;
import com.sap.cloud.security.config.OAuth2ServiceConfigurationBuilder;
import com.sap.cloud.security.config.Service;
import com.sap.cds.repackaged.audit.api.exception.InvalidTokenIssuerException;
import com.sap.xs.audit.message.ValidationError;

public class OAuth2ServiceConfigurationFactory {
    private static final String SUBSCRIBER_TOKEN_ISSUER_IS_INVALID = "Subscriber token issuer is invalid!";
    private static final Logger LOGGER = LoggerFactory.getLogger(OAuth2ServiceConfigurationFactory.class);

    public OAuth2ServiceConfiguration getOAuth2ServiceConfiguration(OAuthCredentials oauthCredentials)
            throws InvalidTokenIssuerException {
        final OAuth2ServiceConfiguration config = oauthCredentials.getCerturl() == null
                ? getOAuth2Config(oauthCredentials)
                : getOAuth2ConfigX509(oauthCredentials);
        return config;
    }

    private OAuth2ServiceConfiguration getOAuth2ConfigX509(OAuthCredentials oauthCredentials)
            throws InvalidTokenIssuerException {
        String certUrl = decideOnUrl(oauthCredentials.getCerturl(), oauthCredentials.getSubscriberTokenIssuer());
        return OAuth2ServiceConfigurationBuilder.forService(Service.XSUAA)
                .withClientId(oauthCredentials.getClientid())
                .withCertUrl(certUrl)
                .withCertificate(oauthCredentials.getCertificate())
                .withPrivateKey(oauthCredentials.getKey())
                .withCredentialType(CredentialType.X509)
                .build();
    }

    private OAuth2ServiceConfiguration getOAuth2Config(OAuthCredentials oauthCredentials)
            throws InvalidTokenIssuerException {
        String url = decideOnUrl(oauthCredentials.getUrl(), oauthCredentials.getSubscriberTokenIssuer());
        return OAuth2ServiceConfigurationBuilder.forService(Service.XSUAA)
                .withClientId(oauthCredentials.getClientid())
                .withClientSecret(oauthCredentials.getClientsecret())
                .withUrl(url)
                .build();
    }

    private String decideOnUrl(String providerTokenIssuer, String subscriberTokenIssuer) throws InvalidTokenIssuerException {
        if (subscriberTokenIssuer != null) {
            LOGGER.debug("Subscriber token issuer is present: {}", subscriberTokenIssuer);
            try {
                validateSubscriberTokenIssuer(providerTokenIssuer, subscriberTokenIssuer);
                return subscriberTokenIssuer;
            } catch (ValidationError e) {
                throw new InvalidTokenIssuerException(INVALID_SUBSCRIBER_TOKEN_ISSUER_ERROR, e);
            }
        }
        return providerTokenIssuer;
    }

    private void validateSubscriberTokenIssuer(String providerTokenIssuer, String subscriberTokenIssuer)
            throws ValidationError {
        String subscriberTokenIssuerDomain = extractDomain(subscriberTokenIssuer);
        String providerTokenIssuerDomain = extractDomain(providerTokenIssuer);

        if (!subscriberTokenIssuerDomain.equals(providerTokenIssuerDomain)) {
            throw new ValidationError(SUBSCRIBER_TOKEN_ISSUER_IS_INVALID);
        }
    }

    private String extractDomain(String tokenIssuer) throws ValidationError {
        try {
            String host = new URI(tokenIssuer).getHost();
            return host.substring(host.indexOf("."));
        } catch (URISyntaxException e) {
            throw new ValidationError(SUBSCRIBER_TOKEN_ISSUER_IS_INVALID);
        }
    }

}
