package com.sap.cds.repackaged.audit.client.impl;

import static com.sap.cds.repackaged.audit.client.impl.Utils.USER_VALUE;

import com.sap.cds.repackaged.audit.api.TransactionalAuditLogMessage;
import com.sap.cds.repackaged.audit.api.exception.AuditLogNotAvailableException;
import com.sap.cds.repackaged.audit.api.exception.AuditLogWriteException;
import com.sap.cds.repackaged.audit.api.v2.AuditLogMessageFactory;
import com.sap.xs.audit.message.ValidationError;
import com.sap.xs.audit.message.utils.JsonParserUtils;
import java.io.IOException;
import java.time.Instant;
import java.util.UUID;

public class TransactionalLogImpl<T extends com.sap.xs.audit.message.TransactionalMessage> extends AuditLogMessageImpl<T> implements TransactionalAuditLogMessage {

	public TransactionalLogImpl(Communicator communicator) {
		super(communicator);
	}
	
	public TransactionalLogImpl(AuditLogMessageFactory factory, Communicator communicator) {
		super(communicator);
	}
	
	@Override
	public void logPrepare() throws AuditLogNotAvailableException, AuditLogWriteException {
		if (alreadyLogged) {
			LOGGER.warn(ALREADY_LOGGED_WARNING);
			return;
		}

		if (message.getUser() == null || USER_VALUE.equals(message.getUser())) {
			String user = Utils.getUser();
			message.setUser(user != null ? user : communicator.getClientId());
		}
		
		if(eventTime == null) {
		    eventTime = Instant.now();
		}
		
		message.setTime(eventTime);
		message.setStatus("BEGIN");

		try {
			message.validate();
		} catch (ValidationError e) {
			LOGGER.error("Audit log message cannot be validated.");
			throw new AuditLogWriteException("Audit log message cannot be validated.", message.getErrors());
		}

		try {
			String resp = communicator.send(serializeMessage(), endpoint, message.getSubscriberTokenIssuer());

			AuditLogIdentifier auditLogIdentifier = JsonParserUtils.desrializeMessage(resp, AuditLogIdentifier.class);

			message.setId(auditLogIdentifier.getId());
			alreadyPrepared = true;
		} catch (IOException e) {
			throw new AuditLogWriteException("Audit server returned invalid response.", e);
		}

	}

	@Override
	public void logSuccess() throws AuditLogNotAvailableException, AuditLogWriteException {
		logStatus(true);
	}

	@Override
	public void logFailure() throws AuditLogNotAvailableException, AuditLogWriteException {
		logStatus(false);
	}
	
	private void logStatus(boolean success) throws AuditLogNotAvailableException, AuditLogWriteException {
		
		if (alreadyLogged) {
			LOGGER.warn(ALREADY_LOGGED_WARNING);
			return;
		}
		
		// generate a unique GUID for the success/non-success message that to be different than the prepare message guid
		message.setUuid(UUID.randomUUID().toString());

		if (message.getUser() == null || USER_VALUE.equals(message.getUser())) {
			String user = Utils.getUser();
			message.setUser(user != null ? user : communicator.getClientId());
		}

		if(eventTime == null) {
		    eventTime = Instant.now();
		}
	        
		message.setTime(eventTime);

		try {
			message.validate();
		} catch (ValidationError e) {
			throw new AuditLogWriteException("Audit log message cannot be validated.", message.getErrors());
		}

		message.setSuccess(success);
		message.setStatus("END");
		communicator.send(serializeMessage(), endpoint, message.getSubscriberTokenIssuer());
		alreadyLogged = true;
	}
}
