package com.sap.cds.repackaged.audit.client.impl;

import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.http.client.HttpClient;
import org.apache.http.impl.client.CloseableHttpClient;

import com.sap.cloud.security.config.ClientIdentity;
import com.sap.cloud.security.config.OAuth2ServiceConfiguration;
import com.sap.cloud.security.xsuaa.client.DefaultOAuth2TokenService;
import com.sap.cloud.security.xsuaa.client.XsuaaDefaultEndpoints;
import com.sap.cloud.security.xsuaa.tokenflows.XsuaaTokenFlows;
import com.sap.cds.repackaged.audit.api.exception.InvalidTokenIssuerException;

public class XsuaaTokenFlowsFactory {
    private static final int MAX_MAP_CAPACITY = 1000;

    private final HttpClient httpClient;
    private final ClientIdentityFactory clientIdentityFactory;
    private final OAuth2ServiceConfigurationFactory configFactory;
    private final Map<String, XsuaaTokenFlows> xsuaaTokenFlowsMap;

    XsuaaTokenFlowsFactory(HttpClient httpClient, ClientIdentityFactory clientIdentityFactory,
            OAuth2ServiceConfigurationFactory configFactory,
            Map<String, XsuaaTokenFlows> tokenFlowsMap) {
        this.httpClient = httpClient;
        this.clientIdentityFactory = clientIdentityFactory;
        this.configFactory = configFactory;
        this.xsuaaTokenFlowsMap = tokenFlowsMap;
    }

    public XsuaaTokenFlowsFactory(HttpClient httpClient) {
        this(httpClient, new ClientIdentityFactoryImpl(), new OAuth2ServiceConfigurationFactory(), createLinedHashMap());
    }

    public XsuaaTokenFlows getXsuaaTokenFlows(OAuthCredentials oauthCredentials)
            throws InvalidTokenIssuerException {
        final String subscriberIssuer = oauthCredentials.getSubscriberTokenIssuer();
        if (subscriberIssuer == null) {
            final String providerIssuer = getProviderIssuer(oauthCredentials);
            return getXsuaaTokenFlows(oauthCredentials, providerIssuer);
        }

        return getXsuaaTokenFlows(oauthCredentials, subscriberIssuer);
    }

    private XsuaaTokenFlows getXsuaaTokenFlows(OAuthCredentials oauthCredentials, String tokenIssuer) throws InvalidTokenIssuerException {
        final XsuaaTokenFlows tokenIssuerTokenFlows = xsuaaTokenFlowsMap.get(tokenIssuer);
        if (tokenIssuerTokenFlows == null) {
            final XsuaaTokenFlows tokenFlows = createXsuaaTokenFlows(oauthCredentials);
            xsuaaTokenFlowsMap.put(tokenIssuer, tokenFlows);
            return tokenFlows;
        }
        return tokenIssuerTokenFlows;
    }

    private String getProviderIssuer(OAuthCredentials oauthCredentials) {
        final String providerIssuer = oauthCredentials.getCerturl();
        return providerIssuer != null ? providerIssuer : oauthCredentials.getUrl();
    }

    private XsuaaTokenFlows createXsuaaTokenFlows(OAuthCredentials oauthCredentials)
            throws InvalidTokenIssuerException {
        final ClientIdentity clientIdentity = clientIdentityFactory.createClientIdentity(oauthCredentials);
        final OAuth2ServiceConfiguration config = configFactory.getOAuth2ServiceConfiguration(oauthCredentials);
        final XsuaaTokenFlows tokenFlows = new XsuaaTokenFlows(
                new DefaultOAuth2TokenService((CloseableHttpClient) httpClient),
                new XsuaaDefaultEndpoints(config),
                clientIdentity);

        return tokenFlows;
    }

    private static Map<String, XsuaaTokenFlows> createLinedHashMap() {
        return new LinkedHashMap<String, XsuaaTokenFlows>() {
            protected boolean removeEldestEntry(Map.Entry<String, XsuaaTokenFlows> eldest) {
                return size() > MAX_MAP_CAPACITY;
            }
        };
    }

}
