package com.sap.cds.repackaged.audit.client.impl.config;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.NoSuchElementException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class ConfigLoaderImpl implements ConfigLoader {

    private static final Logger logger = LoggerFactory.getLogger(com.sap.cds.repackaged.audit.client.impl.config.ConfigLoaderImpl.class);

    private final String directory;

    public ConfigLoaderImpl() {
        this(null);
    }

    public ConfigLoaderImpl(String directory) {
        this.directory = directory;
    }

    @Override
    public String getRequiredEnvironmentVariable(String propertyName) {
        String propertyValue = System.getenv(propertyName);

        if (isBlank(propertyValue)) {
            throw new NoSuchElementException("Environment variable: " + propertyName + " is null or empty");
        }
        return propertyValue;
    }

    @Override
    public String getEnvironmentVariable(String propertyName, String defaultValue) {
        String propertyValue = System.getenv(propertyName);

        if (isBlank(propertyValue)) {
            logger.info("Environment variable: " + propertyName + " is null or empty. Using default value " + defaultValue);
            return defaultValue;
        }
        return propertyValue;
    }

    @Override
    public String getRequiredConfigFromFile(String relativePath) {
        Path path = Paths.get(directory, relativePath);
        try {
            return new String(Files.readAllBytes(path), StandardCharsets.UTF_8).trim();
        } catch (IOException e) {
            throw new RuntimeException("Problem occurred while trying to read config file " + path.toString(), e);
        }
    }

    @Override
    public String getConfigFromFile(String relativePath, String dafaultValue) {
        try {
            return getRequiredConfigFromFile(relativePath);
        } catch (RuntimeException e) {
            logger.info(String.format("Using default value %s", dafaultValue), e);
            return dafaultValue;
        }
    }

    public static boolean isBlank(String str) {
        return (str == null || str.trim().isEmpty());
    }

}
