package com.sap.cds.repackaged.audit.client.impl.v2;

import static com.sap.cds.repackaged.audit.client.impl.Utils.*;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.repackaged.audit.api.exception.AuditLogException;
import com.sap.cds.repackaged.audit.api.v2.AuditLogMessageFactory;
import com.sap.cds.repackaged.audit.api.v2.AuditedDataSubject;
import com.sap.cds.repackaged.audit.api.v2.AuditedObject;
import com.sap.cds.repackaged.audit.api.v2.ConfigurationChangeAuditMessage;
import com.sap.cds.repackaged.audit.api.v2.DataAccessAuditMessage;
import com.sap.cds.repackaged.audit.api.v2.DataModificationAuditMessage;
import com.sap.cds.repackaged.audit.api.v2.SecurityEventAuditMessage;
import com.sap.cds.repackaged.audit.client.impl.Communicator;
import com.sap.cds.repackaged.audit.client.impl.ConsoleCommunicator;
import com.sap.cds.repackaged.audit.client.impl.HttpCommunicator;
import com.sap.xs.env.Credentials;
import com.sap.xs.env.VcapServices;

public class AuditLogMessageFactoryImpl extends AuditLogMessageFactoryImplBase implements AuditLogMessageFactory {

    private static final Logger LOGGER = LoggerFactory.getLogger(LOGGER_NAME);

    static final String AUDIT_SERVICE_URL_PATH_STANDARD_PLAN = AUDIT_SERVICE_URL_PREFIX + "v2/";
    static final String AUDIT_SERVICE_URL_PATH_OAUTH2_PLAN = AUDIT_SERVICE_URL_PREFIX + "oauth2/v2/";
    private Communicator communicator = null;

    public AuditLogMessageFactoryImpl() throws AuditLogException {
        this((VcapServices) null, null);
    }

    public AuditLogMessageFactoryImpl(String bindingName) throws AuditLogException {
        this((VcapServices) null, bindingName);
    }

    public AuditLogMessageFactoryImpl(VcapServices vcapServices) throws AuditLogException {
        this(vcapServices, null);
    }

    public AuditLogMessageFactoryImpl(VcapServices vcapServices, String bindingName) throws AuditLogException {
        Credentials credentials = getCredentials(vcapServices, bindingName);
        if (credentials == null) {
            LOGGER.error("Missing audit log service credentials parameters. Cannot send audit log message to the backend.");
            communicator = new ConsoleCommunicator();
            return;
        } else {
            String plan = getServicePlan(vcapServices, bindingName);
            String urlFromCredentials = credentials.getUrl();
            String auditlogServiceUrl = plan.equals(OAUTH2_PLAN) //
                    ? urlFromCredentials + AUDIT_SERVICE_URL_PATH_OAUTH2_PLAN //
                    : urlFromCredentials + AUDIT_SERVICE_URL_PATH_STANDARD_PLAN; //
            this.communicator = new HttpCommunicator(credentials, plan, auditlogServiceUrl);
        }
    }

    // constructor only for standard plan
    public AuditLogMessageFactoryImpl(String serviceUrl, String serviceUser, String servicePassword) {
        if (serviceUrl == null || serviceUser == null || servicePassword == null) {
            LOGGER.error("Missing audit log service credentials parameters. Cannot send audit log message to the backend.");
            communicator = new ConsoleCommunicator();
            return;
        }
        Credentials credentials = new Credentials();
        credentials.setUser(serviceUser);
        credentials.setPassword(servicePassword);
        credentials.setUrl(serviceUrl);

        this.communicator = new HttpCommunicator(credentials, STANDARD_PLAN, serviceUrl);
    }

    /**
     * This is a constructor for initialization of AuditLogMessageFactory with OAuth2 plan. All
     * required properties must be provided as parameters in the constructor.
     * 
     * @param serviceUrl This is an endpoint of the Auditlog Write API. This must
     *                   be a full path to the endpoint. Example:
     *            https://api.auditlog.cf.{landscape}:8081/audit-log/oauth2/v2/.
     * @param xsuaaClientId This is a client id used to retrieve token.
     * @param xsuaaSecret This is a secret for the {@linkplain xsuaaClientId}.
     * @param xsuaaUrl This is an endpoint of the authorization server used to
     *                 retrieve the token.
     */
    public AuditLogMessageFactoryImpl(final String serviceUrl, final String xsuaaClientId, final String xsuaaSecret,
            final String xsuaaUrl) {
        if (serviceUrl == null || xsuaaClientId == null || xsuaaSecret == null || xsuaaUrl == null) {
            LOGGER.error("Missing audit log service credentials parameters. Cannot send audit log message to the backend.");
            communicator = new ConsoleCommunicator();
            return;
        }
        Credentials credentials = new Credentials();
        credentials.set("uaa", assembleUaaObject(xsuaaClientId, xsuaaSecret, xsuaaUrl));
        credentials.setUrl(serviceUrl);
        this.communicator = new HttpCommunicator(credentials, OAUTH2_PLAN, serviceUrl);
    }

    public AuditLogMessageFactoryImpl(String serviceUrl, String xsuaaClientId, String xsuaaCertUrl, byte[] certificate, byte[] key) {
        if (serviceUrl == null || xsuaaClientId == null || certificate == null || key == null || xsuaaCertUrl == null) {
            LOGGER.error("Missing audit log service credentials parameters. Cannot send audit log message to the backend.");
            communicator = new ConsoleCommunicator();
            return;
        }
        Credentials credentials = new Credentials();
        credentials.set("uaa", assembleUaaObject(xsuaaClientId, xsuaaCertUrl, certificate, key));
        credentials.setUrl(serviceUrl);
        this.communicator = new HttpCommunicator(credentials, OAUTH2_PLAN, serviceUrl);
    }

    //Constructor that allows for custom communicator   
    public AuditLogMessageFactoryImpl(Communicator customCommunicator) {
        if (customCommunicator == null) {
            this.communicator = new ConsoleCommunicator();
            return;
        }
        this.communicator = customCommunicator;
    }

    void setCommunicator(Communicator httpCommunicator) {
        this.communicator = httpCommunicator;
    }

    Communicator getCommunicator() {
        return communicator;
    }

    @Override
    public DataAccessAuditMessage createDataAccessAuditMessage() {
        return new DataAccessMessageImpl(communicator);
    }

    @Override
    public DataModificationAuditMessage createDataModificationAuditMessage() {
        return new DataModificationMessageImpl(communicator);
    }

    @Override
    public ConfigurationChangeAuditMessage createConfigurationChangeAuditMessage() {
        return new ConfigurationChangeMessageImpl(communicator);
    }

    @Override
    public SecurityEventAuditMessage createSecurityEventAuditMessage() {
        return new SecurityEventMessageImpl(communicator);
    }

    @Override
    public AuditedObject createAuditedObject() {
        return new AuditedObjectImpl();
    }

    @Override
    public AuditedDataSubject createAuditedDataSubject() {
        return new AuditedDataSubjectImpl();
    }

}
