using {
  cuid,
  managed
} from '@sap/cds/common';

namespace sap.changelog;

/** ChangeLog primary entity */
@readonly
@cds.autoexpose
@changelog.internal.storage // Related to changelog storage
@changelog.internal.enrich  // Must be enriched with texts
@title: '{i18n>ChangeHistory}'
entity Changes : cuid {

  createdAt: type of managed:createdAt;
  createdBy: type of managed:createdBy;

  // Represents the value that can be used to group the changes
  @UI.Hidden
  @title: '{i18n>Changes.changeLogID}'
  changeLogID: UUID;

  // Qualified name of the root projection from which changes were detected
  @(Common: {
    Text           : rootEntityText,
    TextArrangement: #TextOnly,
  })
  @title: '{i18n>Changes.rootEntity}'
  @changelog.internal.semantics: 'rootEntity'
  rootEntity: String;

  // Configurable "human-readable" ID of the root entity
  @title: '{i18n>Changes.rootIdentifier}'
  @changelog.internal.semantics: 'rootIdentifier'
  rootIdentifier: String;

  // Attribute of the target entity that is changed
  @title: '{i18n>Changes.attribute}'
  @(Common: {
    Text           : attributeText,
    TextArrangement: #TextOnly,
  })
  @changelog.internal.semantics: 'attribute'
  attribute: String;

  @title: '{i18n>Changes.valueChangedFrom}'
  valueChangedFrom: String(5000);

  @title: '{i18n>Changes.valueChangedTo}'
  valueChangedTo: String(5000);

  // Type of the changed attribute
  @title: '{i18n>Changes.valueDataType}'
  valueDataType: String;

  // Configurable "human-readable" ID of the target entity
  @title: '{i18n>Changes.targetIdentifier}'
  @changelog.internal.semantics: 'targetIdentifier'
  targetIdentifier: String;

  // Qualified name of the projection in which changes were detected
  @title: '{i18n>Changes.targetEntity}'
  @(Common: {
    Text           : targetEntityText,
    TextArrangement: #TextOnly,
  })
  @changelog.internal.semantics: 'targetEntity'
  targetEntity: String;

  // JSON representation of the path from the root entity to the target one
  @title: '{i18n>Changes.path}'
  @changelog.internal.semantics: 'path'
  path: String(5000);

  // Type of modification
  @title: '{i18n>Changes.modification}'
  @(Common: {
    Text           : modificationText,
    TextArrangement: #TextOnly,
  })
  @changelog.internal.semantics: 'modification'
  modification: String enum {
    @title: '{i18n>Changes.modification.create}' create;
    @title: '{i18n>Changes.modification.update}' update;
    @title: '{i18n>Changes.modification.delete}' delete;
  };

  // These elements are used to deliver text descriptions from the backend
  @UI.Hidden
  virtual modificationText: String;

  @UI.Hidden
  virtual attributeText: String;

  @UI.Hidden
  virtual targetEntityText: String;

  @UI.Hidden
  virtual rootEntityText: String;
};

// Exclude the aspect content from drafts
@odata.draft.enabled: false
@(UI: {
  PresentationVariant: {
    Visualizations: ['@UI.LineItem'],
    SortOrder     : [{
      Property  : change.createdAt,
      Descending: true
    }]
  },
  LineItem           : [
    {
      Value             : change.modification,
      @HTML5.CssDefaults: {width: '9%'},
      ![@UI.Importance] : #High
    },
    {
      Value             : change.createdAt,
      @HTML5.CssDefaults: {width: '12%'},
      ![@UI.Importance] : #Low
    },
    {
      Value             : change.createdBy,
      @HTML5.CssDefaults: {width: '9%'},
      ![@UI.Importance] : #High
    },
    {
      Value             : change.targetEntity,
      @HTML5.CssDefaults: {width: '11%'},
      ![@UI.Importance] : #High
    },
    {
      Value             : change.targetIdentifier,
      @HTML5.CssDefaults: {width: '14%'},
      ![@UI.Importance] : #Medium
    },
    {
      Value             : change.attribute,
      @HTML5.CssDefaults: {width: '9%'},
      ![@UI.Importance] : #High
    },
    {
      Value             : change.valueChangedTo,
      @HTML5.CssDefaults: {width: '11%'},
      ![@UI.Importance] : #High
    },
    {
      Value             : change.valueChangedFrom,
      @HTML5.CssDefaults: {width: '11%'},
      ![@UI.Importance] : #Medium
    },
    {
      Value            : change.rootIdentifier,
      ![@UI.Importance]: #Low
    },
    {
      Value            : change.rootEntity,
      ![@UI.Importance]: #Low
    },
    {
      Value            : change.path,
      ![@UI.Importance]: #Low
    },
    {
      Value            : change.valueDataType,
      ![@UI.Importance]: #Low
    }
  ],
})
@readonly
@cds.search: {
  change
}
@changelog.internal.enrich
@changelog.internal.storage
aspect ChangeLink {
  @title: '{i18n>Changes.ID}'
  @UI.Hidden
  key change : Association to Changes;
};

/***/
aspect changeTracked {
  @changelog.internal.storage
  changes: Composition of many ChangeLink;
};