/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.changetracking;

import static com.sap.cds.feature.changetracking.tracking.ChangeTrackableElement.IS_RELEVANT;

import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

import com.sap.cds.Result;
import com.sap.cds.Row;
import com.sap.cds.feature.changetracking.tracking.ChangeTracker;
import com.sap.cds.feature.changetracking.tracking.components.ContextParameterHelper;
import com.sap.cds.feature.changetracking.tracking.components.EntityStateReader;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CdsUpsertEventContext;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.services.utils.outbox.OutboxUtils;

// Handlers needs to be executed around existing audit log handlers to override the data for it
@ServiceName(value = "*", type = PersistenceService.class)
public class ChangeTrackingPersistenceServiceHandler implements EventHandler {

	private final EntityStateReader reader = new EntityStateReader(IS_RELEVANT);

	@After
	@HandlerOrder(OrderConstants.After.AUDIT + HandlerOrder.AFTER)
	void afterCreate(CdsCreateEventContext context) {
		// For the creation, we can use the state of the entity that is supplied
		// with the Insert statement
		track(context, context.getResult(), () -> reader.newImage(context));
	}

	@Before
	@HandlerOrder(OrderConstants.Before.ADAPT_STATEMENT + HandlerOrder.BEFORE)
	void beforeUpdate(CdsUpdateEventContext context) {
		// For the update, we need a state of the entity that has all elements that
		// were updated (given that they are annotated with the @changelog annotation)
		// plus primary keys and the elements that are included
		// in the entity identification
		fetch(context, () -> reader.oldImage(context));
	}

	@After
	@HandlerOrder(OrderConstants.After.AUDIT + HandlerOrder.AFTER)
	void afterUpdate(CdsUpdateEventContext context) {
		track(context, context.getResult(), () -> reader.newImage(context));
	}

	@Before
	@HandlerOrder(OrderConstants.Before.ADAPT_STATEMENT + HandlerOrder.BEFORE)
	void beforeUpsert(CdsUpsertEventContext context) {
		// Assume that Upsert is just a simpler version of the Update
		fetch(context, () -> reader.oldImage(context));
	}

	@After
	@HandlerOrder(OrderConstants.After.AUDIT + HandlerOrder.AFTER)
	void afterUpsert(CdsUpsertEventContext context) {
		track(context, context.getResult(), () -> reader.newImage(context));
	}

	@Before
	@HandlerOrder(OrderConstants.Before.ADAPT_STATEMENT + HandlerOrder.BEFORE)
	void beforeDelete(CdsDeleteEventContext context) {
		// Assume that for deletion we need the old state of the entries
		// with all elements that are annotated by the @changelog annotation plus
		// all the primary keys and elements that are included in the entity identification
		fetch(context, () -> reader.oldImage(context));
	}

	@After
	@HandlerOrder(OrderConstants.After.AUDIT + HandlerOrder.AFTER)
	void afterDelete(CdsDeleteEventContext context) {
		// New state is always empty
		track(context, context.getResult(), Collections::emptyList);
	}

	private static void wipeContext(EventContext context) {
		// Try to avoid dangling data if the event context escape the service layer
		ContextParameterHelper.checkInImage(context, null);
		ContextParameterHelper.checkInUpdateShape(context, null);
	}

	private static void fetch(EventContext context, Supplier<List<Row>> image) {
		CdsEntity target = context.getTarget();
		if (isNotTechnicalEntity(target)) {
			List<Row> oldImage = image.get();
			ContextParameterHelper.checkInImage(context, oldImage);
		}
	}

	private static void track(EventContext context, Result result,
		Supplier<List<Row>> image) {
		try {
			if (result != null && result.rowCount() > 0) {
				CdsEntity target = context.getTarget();
				if (isNotTechnicalEntity(target)) {
					List<Row> oldImage = ContextParameterHelper.checkOutImage(context);
					List<Row> newImage = image.get();
					if (!(oldImage.isEmpty() && newImage.isEmpty())) {
						ChangeTracker.apply(context, IS_RELEVANT, newImage, oldImage);
					}
				}
			}
		} finally {
			wipeContext(context);
		}
	}

	private static boolean isNotTechnicalEntity(CdsEntity target) {
		//REVISIT: Once we will be able to overlay the model, we can change this filter to be positive
		//and take only the entities that we will mark with the internal annotation at runtime
		String qualifiedName = target.getQualifiedName();
		return !(qualifiedName.endsWith(Drafts.DRAFT_ADMINISTRATIVE_DATA) || qualifiedName.endsWith("_drafts")
				|| qualifiedName.equals(OutboxUtils.OUTBOX_MODEL)
				|| CdsAnnotations.CHANGELOG_INTERNAL_STORAGE.isTrue(target));
	}
}
