/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.changetracking.tracking;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import com.sap.cds.CdsData;
import com.sap.cds.CdsDataProcessor;
import com.sap.cds.CdsDiffProcessor;
import com.sap.cds.CdsList;
import com.sap.cds.feature.changetracking.Changes;
import com.sap.cds.feature.changetracking.Changes.Modification;
import com.sap.cds.feature.changetracking.tracking.components.IdentifierHelper;
import com.sap.cds.impl.diff.DiffProcessor;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.ql.impl.PathImpl;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsKind;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;
import com.sap.cds.services.EventContext;
import com.sap.cds.util.CdsModelUtils;

@SuppressWarnings("UnstableApiUsage")
public class ChangeTracker implements CdsDiffProcessor.DiffVisitor, CdsDataProcessor.Filter {

	private final Predicate<CdsElement> isRelevant;
	// Path is not 100% safe for the keys (instances are recreated). It is not a problem though
	// as we fetch values from here with a keys only for changed elements, where paths are stable.
	private final Map<Path, List<Changes>> changedElements = new HashMap<>();
	private final Map<Path, List<Changes>> changedAssociations = new HashMap<>();

	public ChangeTracker(Predicate<CdsElement> isRelevant) {
		this.isRelevant = isRelevant;
	}

	public static void apply(EventContext context,
		Predicate<CdsElement> isRelevant,
		Collection<? extends Map<String, Object>> newImage,
		Collection<? extends Map<String, Object>> oldImage) {

		ChangeTracker changeTracker = new ChangeTracker(isRelevant);
		DiffProcessor.create().forDeepTraversal()
			.add(changeTracker, changeTracker)
			.process(newImage, oldImage, context.getTarget());

		ChangeTrackerResultHandler.handle(context, changeTracker.result());
	}

	@Override
	public boolean test(Path path, CdsElement element, CdsType type) {
		// Need to consider: root and all associations as we may need to traverse through them
		return element == null || element.getType().isAssociation()
			|| isChangeTrackedAssociation(path.target().element())
			|| isRelevant.test(element);
	}

	@Override
	public void changed(Path newPath, Path oldPath, CdsElement element, Object newValue, Object oldValue) {
		CdsElement association = newPath.target().element();
		if (element.isKey() && isChangeTrackedAssociation(association) && !IdentifierHelper.elementsOfIdentifier(association).isEmpty()) {
			// We might receive several calls for the same association when its compound key changed
			if (!changedAssociations.containsKey(newPath)) {
				// Association change is bound to the parent entity, not the target of the real path. That is
				// why we recreate the path here. We are within the association here.
				Changes change = ChangesImpl.createFrom(pathToParent(newPath));
				change.setValueDataType(CdsBaseType.STRING.cdsName());
				change.setAttribute(newPath.target().element().getName());
				change.setModification(Modification.UPDATE);
				Object newIdentifier = Objects.requireNonNullElse(IdentifierHelper.getIdentifier(association, newPath.target().values()), newValue);
				Object oldIdentifier = Objects.requireNonNullElse(IdentifierHelper.getIdentifier(association, oldPath.target().values()), oldValue);

				consume(oldIdentifier, change::setValueChangedFrom);
				consume(newIdentifier, change::setValueChangedTo);
				push(changedAssociations, newPath, List.of(change));
			}
		} else {
			Changes change;
			// Assume that DiffProcessor will traverse changed elements one by one without sudden changes of context
			// and that it is safe to calculate most of the context for changed elements once per path that we have seen
			if (changedElements.containsKey(newPath)) {
				Changes template = changedElements.get(newPath).get(0);
				change = ChangesImpl.createFrom(template);
			} else {
				change = ChangesImpl.createFrom(newPath);
				change.setPath(newPath.toRef().toString());
				change.setModification(Modification.UPDATE);
			}
			change.setAttribute(element.getName());
			change.setValueDataType(element.getType().getQualifiedName());
			consume(oldValue, change::setValueChangedFrom);
			consume(newValue, change::setValueChangedTo);
			push(changedElements, newPath, List.of(change));
		}
	}

	@Override
	public void added(Path newPath, Path oldPath, CdsElement association, Map<String, Object> newValue) {
		onAssociationChange(newPath, oldPath, association, newValue, (c, v) -> {
			c.setModification(Modification.CREATE);
			consume(v, c::setValueChangedTo);
		});
	}

	@Override
	public void removed(Path newPath, Path oldPath, CdsElement association, Map<String, Object> oldValue) {
		onAssociationChange(newPath, oldPath, association, oldValue, (c, v) -> {
			c.setModification(Modification.DELETE);
			consume(v, c::setValueChangedFrom);
		});
	}

	private Map<Path, List<Changes>> result() {
		String changeLogId = UUID.randomUUID().toString();
		Map<Path, List<Changes>> result = new HashMap<>();
		Stream.concat(changedElements.entrySet().stream(), changedAssociations.entrySet().stream())
			.forEach(e -> {
				String rootEntity = e.getKey().root().type().getQualifiedName();
				String rootIdentifier = IdentifierHelper.getIdentifier(e.getKey().root());
				e.getValue().forEach(item -> {
					item.setId(UUID.randomUUID().toString());
					item.setRootEntity(rootEntity);
					item.setRootIdentifier(rootIdentifier);
					item.setChangeLogID(changeLogId);
				});
				result.computeIfAbsent(e.getKey(), s -> new ArrayList<>()).addAll(e.getValue());
			});
		return result;
	}

	private void onAssociationChange(Path newPath, Path oldPath, CdsElement association, Map<String, Object> state,
		BiConsumer<Changes, Object> action) {

		if (isChangeTrackedAssociation(association)) {
			String identifier = IdentifierHelper.getIdentifier(association, state);
			if (identifier != null) {
				// Association change belongs to the parent -> no extension here
				Path pathWithValues = selectPathWithValues(newPath, oldPath);
				Changes associationChange = ChangesImpl.createFrom(pathWithValues);
				associationChange.setValueDataType(CdsBaseType.STRING.cdsName());
				associationChange.setAttribute(association.getName());
				action.accept(associationChange, identifier);
				push(changedAssociations, pathWithValues, List.of(associationChange));
			} else {
				// We need to step into the association to generate changes for the keys -> extending the path
				Path extendedPath = extendPath(selectPathWithValues(newPath, oldPath), association, state);
				// Generate changelog for keys
				Set<String> keys = CdsModelUtils.targetKeys(association);
				push(changedElements, extendedPath, handleAssociation(extendedPath, state, e -> keys.contains(e.getName()), action));
			}
		}
		// Traverse into the association content or the new entity state -> path is extended or recreated
		Path extendedPath = extendPath(selectPathWithValues(newPath, oldPath), association, state);
		push(changedElements, extendedPath, handleAssociation(extendedPath, state, isRelevant, action));
	}

	private Path extendPath(Path from, CdsElement association, Map<String, Object> state) {
		// Extend the path to get ref with the association at the end
		if (association != null) {
			// association is null when something is inserted or removed from the root once on that root
			CdsStructuredType target = association.getType().as(CdsAssociationType.class).getTarget();
			return ((PathImpl) from).append(association, target, state);
		} else {
			// Path missing a filter -> extending it to get a proper ref from it
			return new PathImpl(new LinkedList<>()).append(null, from.target().type(), state);
		}
	}

	@SuppressWarnings("unchecked")
	private static void consume(Object value, Consumer<String> setter) {
		if (value != null) {
			if (value instanceof Map map) {
				setter.accept(CdsData.create(map).toJson());
			} else if (value instanceof List collection) {
				setter.accept(CdsList.create(collection).toJson());
			} else {
				setter.accept(value.toString());
			}
		}
	}

	private void push(Map<Path, List<Changes>> result, Path path, List<Changes> changes) {
		if (!changes.isEmpty()) {
			result.computeIfAbsent(path, s -> new ArrayList<>()).addAll(changes);
		}
	}

	private List<Changes> handleAssociation(Path path, Map<String, Object> state,
		Predicate<CdsElement> takeIf, BiConsumer<Changes, Object> action) {

		List<Changes> result = new LinkedList<>();

		Changes template = ChangesImpl.createFrom(path);
		template.setTargetIdentifier(IdentifierHelper.getIdentifier(path.target()));
		template.setPath(path.toRef().toString());
		state.entrySet().stream()
			.filter(e -> e.getValue() != null && path.target().type().findElement(e.getKey()).isPresent())
			.forEach(item -> {
				CdsElement element = path.target().type().getElement(item.getKey());
				if (!element.getType().isAssociation() && takeIf.test(element) &&
					element.getDeclaringType().getKind().equals(CdsKind.ENTITY)) {

					Changes change = ChangesImpl.createFrom(template);
					change.setAttribute(element.getName());
					change.setValueDataType(element.getType().getQualifiedName());

					action.accept(change, item.getValue());
					result.add(change);
				}
			});
		return result;
	}

	private static Path selectPathWithValues(Path newPath, Path oldPath) {
		// In case of the deep deletion, the new path will have segments with empty values at the end
		// The last one is guaranteed to be empty anyway
		return newPath.target().values().isEmpty() ? oldPath : newPath;
	}

	private boolean isChangeTrackedAssociation(CdsElement association) {
		// Association that represents itself as the value defined by the annotation on the association of it
		return (association != null
			&& (association.getType().isAssociation() && CdsModelUtils.isSingleValued(association.getType())
				|| association.getType().as(CdsAssociationType.class).isComposition())
			&& isRelevant.test(association));
	}

	private static Path pathToParent(Path from) {
		//Assume that path always has more than one segment and ends with an association
		return new PathImpl(StreamSupport.stream(from.spliterator(), false)
			.takeWhile(s -> !s.equals(from.target())).collect(Collectors.toCollection(LinkedList::new)));
	}
}
