/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.changetracking.tracking;

import static com.sap.cds.feature.changetracking.tracking.ChangesToPresentationConverter.isChangeLogElement;

import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;

import com.sap.cds.feature.changetracking.Changes;
import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.cqn.AnalysisResult;
import com.sap.cds.ql.cqn.CqnAnalyzer;
import com.sap.cds.ql.cqn.CqnExpand;
import com.sap.cds.ql.cqn.CqnInline;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CdsModelUtils;

@SuppressWarnings("UnstableApiUsage")
@ServiceName(value = "*", type = ApplicationService.class)
public class ChangeViewHandler implements EventHandler {

	@Before(entity = "*")
	@HandlerOrder(OrderConstants.Before.CHECK_CAPABILITIES)
	void checkChangesAccess(CdsReadEventContext context) {
		// Modifying events will be terminated by OData Capabilities annotations if the access has
		// the changelog entity that is not exposed explicitly
		CdsStructuredType target = context.getTarget();
		if (CdsAnnotations.AUTOEXPOSED.isTrue(target) && CdsAnnotations.CHANGELOG_INTERNAL_STORAGE.isTrue(target)) {
			AnalysisResult result = CqnAnalyzer.create(context.getModel()).analyze(context.getCqn());
			CdsStructuredType root = result.rootEntity();
			if (CdsAnnotations.CHANGELOG_INTERNAL_STORAGE.isTrue(root)) {
				throw new ErrorStatusException(CdsErrorStatuses.ENTITY_NOT_READABLE,
						context.getTarget().getQualifiedName());
			}
		}
	}

	@After(entity = "*")
	@HandlerOrder(OrderConstants.After.ADD_FIELDS)
	void extendChangeView(CdsReadEventContext context) {
		if (requiresConversion(context)) {
			// NB: The type of the result is fake, but it does retain the annotations from the elements
			ChangesToPresentationConverter converter = new ChangesToPresentationConverter(context);
			DataProcessor.create()
					.action(new DataProcessor.Action() {
						@Override
						public void entry(Path path, CdsElement element, CdsStructuredType type, Map<String, Object> entry) {
							type.elements()
									.filter(e -> CdsAnnotations.CHANGELOG_INTERNAL_SEMANTICS.getOrValue(e, null) != null
											&& entry.containsKey(e.getName()))
									.forEach(e -> {
										String semantics = CdsAnnotations.CHANGELOG_INTERNAL_SEMANTICS.getOrValue(e, "");
										Optional<String> s = resolveTextElement(e);
										if (s.isPresent()) {
											switch (semantics) {
												case Changes.MODIFICATION -> entry.computeIfAbsent(s.get(),
														k -> converter.convertModification(e, entry.get(e.getName())));
												case Changes.ROOT_ENTITY, Changes.TARGET_ENTITY ->
														entry.computeIfAbsent(s.get(),
																k -> converter.convertEntityName(entry.get(e.getName())));
												case Changes.ATTRIBUTE -> entry.computeIfAbsent(s.get(),
														k -> {
															// To translate the attribute, we need to know the target entity
															// return the value "as-is" if the target entity is not selected
															// or not included in the view
															Optional<CdsElement> targetEntity = type.elements()
																	.filter(t -> CdsAnnotations.CHANGELOG_INTERNAL_SEMANTICS.getOrValue(t, "")
																			.equals(Changes.TARGET_ENTITY)).findFirst();
															if (targetEntity.isPresent()) {
																return converter.convertAttribute(
																		entry.get(targetEntity.get().getName()),
																		entry.get(e.getName()));
															}
															return entry.get(e.getName());
														});
												default -> {
													// leave the rest of the elements unchanged
												}
											}
										}
									});
						}
					})
					.addConverter(
							(path, element, type) -> isChangeLogElement(element, Changes.PATH),
							(path, element, value) -> {
								if (value instanceof String string) {
									return converter.convertPath(string);
								}
								return value;
							})
					.process(context.getResult());
		}
	}

	private static Optional<String> resolveTextElement(CdsElement element) {
		return Optional.ofNullable(CdsAnnotations.COMMON_TEXT
				.getOrValue(element, Map.of()).get("=")).map(Object::toString);
	}

	private static boolean requiresConversion(CdsReadEventContext context) {
		CdsStructuredType target = context.getTarget();
		Object value = CdsAnnotations.CHANGELOG_INTERNAL_ENRICH.getOrValue(target, null);
		if (value != null) {
			return Boolean.TRUE.equals(value);
		} else {
			// This visitor must be target-aware and handle deep expands (changes from one entity requested from other entity)
			// Also inline(). It adds complexity as the inline has simple ref pointing to the part of the path of original target.
			TargetsAssociation targetsAssociation =
					new TargetsAssociation(CdsAnnotations.CHANGELOG_INTERNAL_ENRICH::isTrue, context.getTarget());
			context.getCqn().accept(targetsAssociation);
			return targetsAssociation.getResult();
		}
	}

	private static class TargetsAssociation implements CqnVisitor {

		private final Predicate<CdsStructuredType> condition;
		private final CdsStructuredType target;
		private boolean result = false;

		private TargetsAssociation(Predicate<CdsStructuredType> condition, CdsStructuredType target) {
			this.condition = condition;
			this.target = target;
		}

		@Override
		public void visit(CqnInline inline) {
			if (!result) {
				CdsStructuredType assocTarget = CdsModelUtils.target(target, inline.ref().segments());
				if (condition.test(assocTarget)) {
					result = true;
				}
			}
		}

		@Override
		public void visit(CqnExpand expand) {
			// Do not enrich the view if the query contains a star expand:
			// it will not expand the changeLog association anyway and this is actually quite expensive
			// to traverse the data to find the elements of the changelog
			if (!result && (!expand.ref().rootSegment().id().equals("*"))) {
				CdsStructuredType assocTarget = CdsModelUtils.target(target, expand.ref().segments());
				if (condition.test(assocTarget)) {
					result = true;
				}
				expand.items().forEach(i -> {
					if (!result) {
						TargetsAssociation targetsAssociation =
								new TargetsAssociation(this.condition, assocTarget);
						i.accept(targetsAssociation);
						result = targetsAssociation.getResult();
					}
				});
			}
		}

		public boolean getResult() {
			return result;
		}
	}
}
