/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.changetracking.tracking;

import com.sap.cds.adapter.edmx.EdmxI18nProvider;
import com.sap.cds.impl.parser.JsonParser;
import com.sap.cds.impl.parser.TokenParser;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEnumType;
import com.sap.cds.reflect.CdsEnumType.Enumeral;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.utils.model.CdsAnnotations;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

class ChangesToPresentationConverter {

	private static final Logger logger = LoggerFactory.getLogger(ChangesToPresentationConverter.class);
	private static final Pattern I18N_KEY = Pattern.compile("\\{i18n>(.+)\\}");
	private final CdsModel model;
	private final Map<String, String> texts;

	ChangesToPresentationConverter(EventContext context) {
		this.model = context.getModel();
		EdmxI18nProvider provider = context.getCdsRuntime().getProvider(EdmxI18nProvider.class);
		if (provider != null) {
			this.texts = provider.getTexts(context.getParameterInfo().getLocale());
		} else {
			this.texts = Collections.emptyMap();
		}
	}

	public Object convertModification(CdsElement element, Object value) {
		if (value instanceof String string) {
			CdsEnumType<?> cdsEnumType = element.getType().as(CdsEnumType.class);
			Map<String, ? extends Enumeral<?>> enumItems = cdsEnumType.enumerals();
			if (enumItems.containsKey(string)) {
				return getTitleOrDefault(enumItems.get(string), string);
			}
		}
		return value;
	}

	public Object convertEntityName(Object value) {
		if (value instanceof String string) {
			return model.findEntity(string).map(e -> getTitleOrDefault(e, string)).orElse(string);
		}
		return value;
	}

	public Object convertAttribute(Object entity, Object value) {
		//NB: In the default annotations of this view delivered by us, the value should always present
		if (entity instanceof String entityName && value instanceof String attributeName) {
			return model.findEntity(entityName)
					.flatMap(e -> e.findElement(attributeName))
					.map(element -> getTitleOrDefault(element, attributeName))
					.orElse(attributeName);
		}
		return value;
	}

	public Object convertPath(Object value) {
		if (value instanceof String path) {
			try {
				StructuredTypeRef ref = TokenParser.ref(JsonParser.parseJson(path));
				return model.findEntity(ref.rootSegment().id())
						.map(r -> RefSerializer.serialize(ref)).orElse(path);
			} catch (Exception e) {
				logger.warn("Failed to parse value of the path in the changelog, the original value will be returned");
			}
		}
		return value;
	}

	private String getTitleOrDefault(CdsAnnotatable from, String defaultValue) {
		String title = CdsAnnotations.TITLE.getOrValue(from, CdsAnnotations.COMMON_LABEL.getOrValue(from, defaultValue));
		if (!texts.isEmpty()) {
			Matcher matcher = I18N_KEY.matcher(title);
			if (matcher.matches()) {
				String key = matcher.group(1);
				return texts.getOrDefault(key, defaultValue);
			} else {
				return title;
			}
		}
		return defaultValue;
	}

	public static boolean isChangeLogElement(CdsElement element, String attribute) {
		return CdsAnnotations.CHANGELOG_INTERNAL_SEMANTICS.getOrDefault(element).equals(attribute);
	}
}
