/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.changetracking.tracking;

import com.sap.cds.ql.cqn.*;
import com.sap.cds.ql.cqn.CqnComparisonPredicate.Operator;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.services.draft.Drafts;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

final class RefSerializer {

	private RefSerializer() {
		// no instances
	}

	public static String serialize(CqnStructuredTypeRef ref) {
		if (ref == null || ref.segments().isEmpty()) {
			return "";
		}

		List<Segment> segments = ref.segments();
		List<String> target = new ArrayList<>(ref.segments().size());

		for (Segment segment : segments) {
			String result = segment.id();
			// NB: The ref might not correctly preserve the order of the keys
			Optional<CqnPredicate> filter = segment.filter();
			if (filter.isPresent()) {
				FilterSerializer serializer = new FilterSerializer();
				filter.get().accept(serializer);
				if (!serializer.isEmpty()) {
					result += serializer.getResult();
				}
			}
			target.add(result);
		}
		return String.join("/", target);
	}

	private static class FilterSerializer implements CqnVisitor {

		private final List<String> filterString = new ArrayList<>(3);
		private final List<String> draftKeys = new ArrayList<>(1);

		@Override
		public void visit(CqnComparisonPredicate predicate) {
			if (predicate.operator() == Operator.EQ || predicate.operator() == Operator.IS) {
				CqnValue left = predicate.left();
				CqnValue right = predicate.right();
				CqnValue value = null;
				String key = null;
				// Assume only key = 'value' variant
				if (left instanceof CqnElementRef leftRef) {
					key = leftRef.firstSegment();
					value = right;
				}
				if (key != null) {
					Object theValue = null;
					if (value instanceof CqnLiteral<?> literal) {
						theValue = literal.value();
						// Ignore the UUID for practical reason -> it is somewhat expensive to guess one
						// without model information
						if (theValue instanceof String stringValue) {
							theValue = "'" + stringValue + "'";
						}
					}
					if (theValue != null) {
						if (key.equals(Drafts.IS_ACTIVE_ENTITY)) {
							draftKeys.add(0, key + "=" + theValue);
						} else {
							filterString.add(key + "=" + theValue);
						}
					}
				}
			}
		}

		public boolean isEmpty() {
			return filterString.isEmpty();
		}

		public String getResult() {
			filterString.addAll(draftKeys);
			return "(" + String.join(",", filterString) + ")";
		}
	}
}
