/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.changetracking.tracking.components;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import com.google.common.collect.Maps;
import com.sap.cds.Result;
import com.sap.cds.Row;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSortSpecification.Order;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CdsUpsertEventContext;
import com.sap.cds.services.cds.CqnService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.utils.model.CqnUtils;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.CdsModelUtils.CascadeType;
import com.sap.cds.util.DataUtils;

public final class EntityStateReader {

	// Which elements are change tracked?
	private final Predicate<CdsElement> isChangeTracked;

	// Which associations we should cascade to get all change tracked elements?
	private final Predicate<CdsElement> isCascadedForInsert;
	private final Predicate<CdsElement> isCascadedForUpsert;
	private final Predicate<CdsElement> isCascadedForUpdate;
	private final Predicate<CdsElement> isCascadedForDelete;

	public EntityStateReader(Predicate<CdsElement> isChangeTracked) {
		this.isChangeTracked = isChangeTracked;
		this.isCascadedForInsert = (e -> CdsModelUtils.isCascading(CascadeType.INSERT, e));
		this.isCascadedForUpsert = (e -> CdsModelUtils.isCascading(CascadeType.UPDATE, e) || CdsModelUtils.isCascading(CascadeType.INSERT, e));
		this.isCascadedForUpdate = (e -> CdsModelUtils.isCascading(CascadeType.UPDATE, e));
		this.isCascadedForDelete = (e -> CdsModelUtils.isCascading(CascadeType.DELETE, e));
	}

	public List<Row> oldImage(CdsUpdateEventContext context) {
		// For updates, the old image will contain the same data that are present in the data of the statement
		// The values of the elements might be null, but they are not important as they will be selected anyway

		Collection<? extends Map<String, Object>> source =
				ShapeResolver.resolve(context.getTarget(), context.getCqn(), context.getCqnValueSets());
		ContextParameterHelper.checkInUpdateShape(context, source); // We can do this once per statement
		return selectImageForUpdate(context, source);
	}

	public List<Row> oldImage(CdsUpsertEventContext context) {
		// For upserts, the old image will contain the same data that are present in the data of the statement
		// and expand of all tracked elements of associations that targets change tracked elements

		return context.getCqn().entries().stream().flatMap(r ->
			readImage(context, CqnUtils.toSelect(context.getCqn(), context.getTarget()), isChangeTracked, isCascadedForUpsert, r)).toList();
	}

	public List<Row> oldImage(CdsDeleteEventContext context) {
		// For deletions, the old image will contain the change tracked elements across the entity
		// and all the associations that are cascading

		SelectionSetBuilder selectionSetBuilder = new SelectionSetBuilder(context.getTarget(), isChangeTracked, isCascadedForDelete);

		List<CqnSelectListItem> trackedShape = selectionSetBuilder.get();
		if (!trackedShape.isEmpty()) {
			CqnSelect select = CqnUtils.toSelect(context.getCqn()).columns(trackedShape);
			if (context.getCqnValueSets().iterator().hasNext()) {
				return StreamSupport.stream(context.getCqnValueSets().spliterator(), false)
					.flatMap(p -> context.getService().run(select, p).stream()).toList();
			} else {
				return context.getService().run(select).list();
			}
		}
		return Collections.emptyList();
	}

	public List<Row> newImage(CdsCreateEventContext context) {
		// Insert needs an additional fetch, because we need identifiers (values of @changelog annotation)
		// that can point outside the entity e.g. to its associations

		Select<?> statement = Select.from(context.getCqn().ref());
		return context.getResult().stream().flatMap(r -> readImage(context, statement, isChangeTracked, isCascadedForInsert, r)).toList();
	}

	public List<Row> newImage(CdsUpdateEventContext context) {
		Collection<? extends Map<String, Object>> source = ContextParameterHelper.checkOutUpdateShape(context);
		return selectImageForUpdate(context, source);
	}

	public List<Row> newImage(CdsUpsertEventContext context) {
		return context.getResult().stream().flatMap(r ->
			readImage(context, CqnUtils.toSelect(context.getCqn(), context.getTarget()), isChangeTracked, isCascadedForUpsert, r)).toList();
	}

	private List<Row> selectImageForUpdate(CdsUpdateEventContext context, Collection<? extends Map<String, Object>> source) {
		Stream<? extends Map<String, Object>> entries = source.isEmpty() ? context.getCqn().entries().stream() : source.stream();
		return entries.flatMap(r ->
						readImage(context, CqnUtils.toSelect(context.getCqn(), context.getTarget()),
							isChangeTracked, isCascadedForUpdate, DataUtils.copyMap(r)))
				.toList();
	}

	private static Stream<Row> readImage(EventContext context, Select<?> statement,
		Predicate<CdsElement> isChangeTracked, Predicate<CdsElement> isCascaded,
		Map<String, Object> row) {

		// This reader always need to have a row that contains the keys of the entity and the elements that
		// were changed without the actual values (they might be set to null) as they will be selected anyway
		// row must only tell us what elements to select and which associations to expand

		Set<String> keys = CdsModelUtils.keyNames(context.getTarget());
		SelectionSetBuilder selectionSetBuilder = new SelectionSetBuilder(context.getTarget(), isChangeTracked, isCascaded);
		List<CqnSelectListItem> trackedColumns = selectionSetBuilder.get(row);
		if (trackedColumns.isEmpty()) {
			return Stream.empty();
		} else {
			Map<String, Object> matchingKeys = Maps.filterKeys(row, keys::contains);
			if (matchingKeys.isEmpty()) {
				statement = statement.columns(trackedColumns);
			} else {
				statement = statement.columns(trackedColumns).matching(matchingKeys);
			}
			statement.orderBy(keys.stream().map(k -> CQL.sort(CQL.get(k), Order.ASC)).toList());
			CqnSelect selectStatement = statement.asSelect();

			// The update privilege does not imply that one can read the data, also something can be changed
			// by a bulk statement that implicitly targets entities that are not visible to the user, but
			// can be change-tracked
			return context.getCdsRuntime().requestContext()
					.privilegedUser()
					.modifyParameters(c -> c.setLocale(null))
					.run((Function<RequestContext, Result>) rc ->
						((CqnService) context.getService()).run(selectStatement)).stream();
		}
	}
}
