/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.changetracking.tracking.components;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;

import com.sap.cds.feature.changetracking.tracking.SensitiveElement;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.cqn.ResolvedSegment;
import com.sap.cds.reflect.CdsAnnotatable;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.DataUtils;

public final class IdentifierHelper {

	private static final String READABLE_ID = "@ctid:";

	private IdentifierHelper() {
		// hidden
	}

	public static String toIdentifierMarker(String id) {
		return READABLE_ID + StringUtils.replace(id, ".", "_");
	}

	public static List<ElementRef<Object>> elementsOfIdentifier(CdsAnnotatable type) {
		Object value = CdsAnnotations.CHANGELOG.getOrValue(type, null);
		if (value instanceof Boolean) {
			return List.of();
		} else {
			// Annotation may define the elements that define content of the identifier
			// For structured type: own elements or something reachable from the entity via path
			// For elements: elements of the target entity prefixed with an association name

			List<String> members = CdsAnnotations.CHANGELOG.getOrValue(type, List.<Map<String, Object>>of())
				.stream()
				.flatMap(v -> v.values().stream().map(Object::toString))
				.filter(e -> !Drafts.ELEMENTS.contains(e))
				.toList();

			List<ElementRef<Object>> result = new LinkedList<>();
			CdsStructuredType target;
			String prefix = null;
			if (type instanceof CdsElement def) {
				prefix = def.getName();
				target = def.getDeclaringType();
			} else {
				target = (CdsStructuredType) type;
			}

			for (String k : members) {
				ElementRef<Object> ref = CQL.get(k);
				// If prefix exists, it must be the first segment of the ref -> for elements
				if (prefix == null || prefix.equals(ref.firstSegment())) {
					Optional<CdsElement> element = target.findElement(k);
					if (element.isPresent() && !SensitiveElement.IS_SENSITIVE.test(element.get())) {
						result.add(ref);
					}
				}
			}
			return result;
		}
	}

	public static String getIdentifier(ResolvedSegment segment) {
		Stream<Object> source = elementsOfIdentifier(segment.type()).stream()
			.map(e -> DataUtils.getOrDefault(segment.values(), toIdentifierMarker(e.path()), null));
		return stringify(source);
	}

	public static String getIdentifier(CdsElement element, Map<String, Object> values) {
		List<ElementRef<Object>> elementRefs = elementsOfIdentifier(element);
		if (elementRefs.isEmpty()) {
			return null;
		} else {
			Stream<Object> source = elementRefs.stream()
				.map(v -> CQL.get(v.segments().stream().skip(1).toList()))
				.map(e -> DataUtils.getOrDefault(values, toIdentifierMarker(e.path()), null));
			return stringify(source);
		}
	}

	private static String stringify(Stream<Object> stream) {
		return stream.filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(", "));
	}
}
