/**************************************************************************
 * (C) 2019-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.changetracking.tracking.components;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import com.sap.cds.CdsList;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.ElementRef;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.CqnComparisonPredicate.Operator;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsArrayedType;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsVisitor;
import com.sap.cds.services.draft.Drafts;
import com.sap.cds.services.utils.model.CdsAnnotations;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.DataUtils;

public class SelectionSetBuilder {

	private final CdsStructuredType root;
	private final Predicate<CdsElement> isChangeTracked;
	private final Predicate<CdsElement> isCascaded;

	/**
	 * Create an instance of the builder that will return the select list for the given type that
	 * can be used to fetch the image of the data for change tracking.
	 *
	 * @param root            The assessed type
	 * @param isChangeTracked the predicate that must return true if the element is change-tracked
	 * @param isCascaded      the predicate that controls the way the associations are cascaded during search
	 */
	public SelectionSetBuilder(CdsStructuredType root, Predicate<CdsElement> isChangeTracked, Predicate<CdsElement> isCascaded) {
		this.root = root;
		this.isChangeTracked = isChangeTracked;
		this.isCascaded = withoutAssociationToChangeLog(isCascaded);
	}

	/**
	 * Generates the select list that contains change-tracked elements of the given type
	 * that can be used to read an image for it.
	 * The change-tracked elements are represented as a simple items,
	 * associations that are change-tracked or contain change-tracked elements inside them are represented as
	 * potentially nested expands.
	 * The select list always includes the primary keys and adds the identifiers (elements defined by the @changelog annotation)
	 * for the entities (as an aliased elements) and the associations (as an aliased members of the expands together with primary keys).
	 *
	 * @return The select list representing all change-tracked elements of the given type
	 */
	public List<CqnSelectListItem> get() {
		// Build the selection shape using the entity type
		ChangeTrackedElementsFromType elementVisitor =
			new ChangeTrackedElementsFromType(isChangeTracked, isCascaded, Set.of(root.getQualifiedName()));
		root.accept(elementVisitor);
		return elementVisitor.getColumns();
	}

	/**
	 * Generates the select list that represents the change-tracked elements of the given type that
	 * can be used to read an image for it using the provided state of the entity as a filter.
	 * It does not consider the values of the elements (they can and should be null) except
	 * the primary keys.
	 * The parameters and the expressions are not supported and must be resolved beforehand.
	 * For to-many associations, structure of individual data items is not considered,
	 * the expand for it will always contain all change-tracked elements.
	 * For delta-list, the filter will be added to the expand to select only affected items.
	 * The select list always includes the primary keys and adds the identifiers (elements defined by the @changelog annotation)
	 * for the entities (as an aliased elements) and the associations (as an aliased members of the expands together with primary keys).
	 *
	 * @param data the state of the entity
	 * @return The select list representing all change-tracked elements of the given type
	 */
	public List<CqnSelectListItem> get(Map<String, Object> data) {
		return selectColumns(root, data, Set.of(root.getQualifiedName()));
	}

	private List<CqnSelectListItem> selectColumns(CdsStructuredType type, Map<String, Object> data, Set<String> visitedTypes) {
		List<CqnSelectListItem> columns = new LinkedList<>();

		// Take all elements that are present in the data, save for associations encountered
		// in either structured form or as OData foreign key.
		Set<CdsElement> assocElements = new HashSet<>();
		for (String item : data.keySet()) {
			CdsElement element = type.getElement(item);
			if (element.getType().isAssociation() && (isCascaded.test(element) || isChangeTracked.test(element))) {
				assocElements.add(element);
			} else if (isChangeTracked.test(element)) {
				String assoc = CdsAnnotations.ODATA_FOREIGN_KEY_FOR.getOrDefault(element);
				if (assoc != null) {
					assocElements.add(type.getElement(assoc));
				} else if (!element.isKey()) { // To be consistent with the type-based expand
					columns.add(CQL.get(element.getName()));
				}
			}
		}

		// Associations are always added in a structured form either as an expand with identifier elements
		// (values of the elements from the @changelog annotation) or as structured key expand (CQL.get("assoc"))
		// if the association has only the @changelog annotation
		assocElements.stream()
			.map(assocElement -> expandAssociation(assocElement, data, visitedTypes))
			.forEach(columns::addAll);

		if (!columns.isEmpty()) {
			addEntityIdentifier(type, columns);
		}
		return columns;
	}

	private List<CqnSelectListItem> expandAssociation(CdsElement element, Map<String, Object> data, Set<String> visitedTypes) {

		List<CqnSelectListItem> result = new ArrayList<>();

		CdsEntity target = element.getType().as(CdsAssociationType.class).getTarget();
		// With data, we can go deeply into the associations as the data defines finite set of levels
		// for the same entity. This is required to keep the tracking behaviour in sync
		// with type-based implementation where this leads to infinite loop.
		if (visitedTypes.contains(target.getQualifiedName())) {
			return List.of();
		} else {
			Set<String> next = new HashSet<>(visitedTypes);
			next.add(target.getQualifiedName());
			if (CdsModelUtils.isSingleValued(element.getType())) {
				result.addAll(expandToOneAssociation(element, data, next));
			} else {
				result.addAll(expandToManyAssociation(element, data, next));
			}
			return result;
		}
	}

	private List<CqnSelectListItem> expandToOneAssociation(CdsElement element, Map<String, Object> data, Set<String> visitedTypes) {

		CdsEntity target = element.getType().as(CdsAssociationType.class).getTarget();

		Map<String, Object> content = DataUtils.getOrDefault(data, element.getName(), Map.of());
		List<CqnSelectListItem> selectList = new LinkedList<>();

		if (isCascaded.test(element)) {
			if (content != null) {
				List<CqnSelectListItem> expandedColumns = selectColumns(target, content, visitedTypes);
				if (!expandedColumns.isEmpty()) {
					selectList.addAll(expandedColumns);
				}
			} else {
				ChangeTrackedElementsFromType elementVisitor = new ChangeTrackedElementsFromType(isChangeTracked, isCascaded, visitedTypes);
				target.accept(elementVisitor);
				if (!elementVisitor.getColumns().isEmpty()) {
					selectList.addAll(elementVisitor.getColumns());
				}
			}
		}

		List<ElementRef<Object>> identifierElements = IdentifierHelper.elementsOfIdentifier(element);
		if (identifierElements.isEmpty() && selectList.isEmpty() && isChangeTracked.test(element)) {
			return List.of(CQL.get(element.getName()));
		} else {
			CdsModelUtils.targetKeys(element).stream().map(CQL::get).forEach(selectList::add);
			addElementIdentifier(element, selectList);
			return List.of(CQL.to(element.getName()).expand(selectList));
		}
	}

	private List<CqnSelectListItem> expandToManyAssociation(CdsElement element, Map<String, Object> data, Set<String> visitedTypes) {

		CdsEntity target = element.getType().as(CdsAssociationType.class).getTarget();

		List<CqnSelectListItem> selectList = new ArrayList<>();
		// Identifier is considered only for to-many compositions
		if (element.getType().as(CdsAssociationType.class).isComposition()) {
			List<ElementRef<Object>> identifier = IdentifierHelper.elementsOfIdentifier(element);
			if (!identifier.isEmpty()) {
				addElementIdentifier(element, selectList);
			}
		}

		// We may have elements that are change-tracked deeper in the composition items
		ChangeTrackedElementsFromType elementVisitor = new ChangeTrackedElementsFromType(isChangeTracked, isCascaded, visitedTypes);
		target.accept(elementVisitor);
		selectList.addAll(elementVisitor.getColumns());

		if (!selectList.isEmpty()) {
			Set<String> keys = CdsModelUtils.concreteKeyNames(target);

			List<Map<String, Object>> content = DataUtils.getOrDefault(data, element.getName(), List.of());

			// Delta list reduces the number of entries that we need
			CqnPredicate filter = toFilter(content, keys);

			StructuredType<?> path = CQL.to(element.getName());
			if (filter != null) {
				path = path.filter(filter);
			}
			CdsModelUtils.targetKeys(element).stream().map(CQL::get).forEach(selectList::add);
			return List.of(path.expand(selectList));
		}
		return List.of();
	}

	private static CqnPredicate toFilter(List<Map<String, Object>> content, Set<String> keys) {
		if (content instanceof CdsList<Map<String, Object>> cdsList && !cdsList.isEmpty() && cdsList.isDelta()) {
			List<CqnValue> values = new ArrayList<>(cdsList.size());
			List<ElementRef<Object>> references = keys.stream().map(CQL::get).toList();
			cdsList.forEach(i -> keys.forEach(k -> values.add(CQL.val(i.get(k)))));
			return CQL.comparison(CQL.list(references), Operator.EQ, CQL.list(values));
		}
		return null;
	}

	private static class ChangeTrackedElementsFromType implements CdsVisitor {

		private final Predicate<CdsElement> isChangeTracked;
		private final Predicate<CdsElement> isCascaded;
		private final List<CqnSelectListItem> columns = new LinkedList<>();
		private final Set<String> visitedTypes;

		private ChangeTrackedElementsFromType(Predicate<CdsElement> isChangeTracked, Predicate<CdsElement> isCascaded, Set<String> visitedTypes) {
			this.isChangeTracked = isChangeTracked;
			this.isCascaded = isCascaded;
			this.visitedTypes = visitedTypes;
		}

		@Override
		public void visit(CdsEntity entity) {
			// If there is no tracked elements here, no need to add keys and human-readable IDs
			if (!columns.isEmpty()) {
				addEntityIdentifier(entity, columns);
			}
		}

		@Override
		public void visit(CdsElement element) {
			if (!element.getType().isAssociation() && !element.isKey()
				// Prefer the structured expand to the foreign keys
				&& CdsAnnotations.ODATA_FOREIGN_KEY_FOR.getOrDefault(element) == null) {

				if (element.getType().isSimple() && isChangeTracked.test(element)) {
					columns.add(CQL.get(element.getName()));
				} else {
					RelevancyChecker checker = new RelevancyChecker(isChangeTracked);
					element.getType().accept(checker);
					if (checker.getResult()) {
						columns.add(CQL.get(element.getName()));
					}
				}
			} else if (element.getType().isAssociation() && (isCascaded.test(element) || isChangeTracked.test(element))) {
				onAssociation(element);
			}

		}

		private void onAssociation(CdsElement element) {
			// Precaution: if association targets changelog entity, do not go deeper.
			if (CdsAnnotations.CHANGELOG_INTERNAL_STORAGE.isTrue(element)) {
				return;
			}
			List<CqnSelectListItem> selectList = new ArrayList<>();
			CdsAssociationType associationType = element.getType().as(CdsAssociationType.class);
			if (!visitedTypes.contains(associationType.getTarget().getQualifiedName())) {
				if (isCascaded.test(element)) {
					Set<String> next = new HashSet<>(visitedTypes);
					next.add(associationType.getTarget().getQualifiedName());
					ChangeTrackedElementsFromType elementVisitor = new ChangeTrackedElementsFromType(isChangeTracked, isCascaded, next);
					associationType.getTarget().accept(elementVisitor);
					selectList.addAll(elementVisitor.getColumns());
				}
				if (isChangeTracked.test(element) && (associationType.isComposition() || CdsModelUtils.isSingleValued(associationType))) {
					if (selectList.isEmpty()) {
						CdsModelUtils.targetKeys(element).stream().map(CQL::get).forEach(selectList::add);
					}
					addElementIdentifier(element, selectList);
					if (!selectList.isEmpty()) {
						columns.add(CQL.to(element.getName()).expand(selectList));
					} else {
						columns.add(CQL.get(element.getName()));
					}
				} else if (!selectList.isEmpty()) {
					columns.add(CQL.to(element.getName()).expand(selectList));
				}
			}
		}

		List<CqnSelectListItem> getColumns() {
			return columns;
		}

		// Utility: returns true if the given relevancy predicate is true not only for the
		// type of the element, but also across the members of its structured type or an arrayed types in it.
		private static class RelevancyChecker implements CdsVisitor {

			private boolean result = false;
			private final Predicate<CdsElement> isRelevant;

			private RelevancyChecker(Predicate<CdsElement> isRelevant) {
				this.isRelevant = isRelevant;
			}

			private boolean getResult() {
				return result;
			}

			@Override
			public void visit(CdsArrayedType type) {
				type.getItemsType().accept(this);
			}

			@Override
			public void visit(CdsElement element) {
				if (!result) {
					result = isRelevant.test(element);
				}
			}
		}
	}

	private static void addEntityIdentifier(CdsStructuredType entity, List<CqnSelectListItem> columns) {
		//REVISIT: Extract this to make this generic

		// For path and comparison full primary key required (even for drafts).
		entity.keyElements().forEach(k -> columns.add(CQL.get(k.getName())));

		// Human-readable ID from the elements in annotation @changelog
		// Extract and check values in annotation @changelog to use them as human-readable ID
		IdentifierHelper.elementsOfIdentifier(entity).stream()
			.map(v -> v.as(IdentifierHelper.toIdentifierMarker(v.path())))
			.forEach(columns::add);
	}

	private static void addElementIdentifier(CdsElement element, List<CqnSelectListItem> columns) {
		// Human-readable ID from the elements in annotation @changelog on the associations.
		// Members of the human-readable value always assumed to be prefixed with an association name
		// as they are defined on it e.g. [book.name, book.isbn]
		IdentifierHelper.elementsOfIdentifier(element).stream()
			.filter(r -> element.getName().equals(r.firstSegment()))
			.map(v -> CQL.get(v.segments().stream().skip(1).toList()))
			.map(v -> v.as(IdentifierHelper.toIdentifierMarker(v.path())))
			.forEach(columns::add);
	}

	private Predicate<CdsElement> withoutAssociationToChangeLog(Predicate<CdsElement> predicate) {
		return predicate
			// No need to handle the draft elements, they are useless for us
			.and(e -> !Drafts.ELEMENTS.contains(e.getName()))
			// DEFENSIVE: Entity can have association to its changes -> even if this association
			// is modelled wrong, we never will touch the changelog content here
			.and(e -> {
				if (e.getType().isAssociation()) {
					return !CdsAnnotations.CHANGELOG_INTERNAL_STORAGE.isTrue(e.getType().as(CdsAssociationType.class).getTarget());
				}
				return true;
			});
	}
}
