/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.changetracking.tracking.components;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.StreamSupport;

import com.sap.cds.impl.DataProcessor;
import com.sap.cds.ql.cqn.CqnExpression;
import com.sap.cds.ql.cqn.CqnParameter;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.util.DataUtils;

public class ShapeResolver {

	private ShapeResolver() {
		//hidden
	}

	/**
	 * INTERNAL: For the update statement, we need to construct the map that represents the "shape" of the update.
	 * It includes the elements that are used as statement parameters or in setters _without_ the guaranteed values
	 * (they are unknown in general).
	 * This shape can be empty, that means that the update statement has everything inside the data.
	 * The parameters within the data of the entity are not supported, in general.
	 *
	 * @param target     target entity of the statement
	 * @param statement  the update statement
	 * @param parameters the parameters of the statement
	 * @return a list of maps that represents the shape of the update for each entity instance involved
	 */
	public static List<Map<String, Object>> resolve(CdsEntity target, CqnUpdate statement, Iterable<Map<String, Object>> parameters) {
		if (parameters.iterator().hasNext()) {
			return resolveParameters(statement, target, parameters);
		} else if (!statement.setters().isEmpty()) {

			// Setters resolved to the elements they change -> the values are not important
			return statement.entries().stream().map(r -> {
				Map<String, Object> copy = DataUtils.copyMap(r);
				statement.setters().forEach((key, cqnValue) -> DataUtils.putPath(copy, key, null));
				return copy;
			}).toList();
		}
		// shape is defined by the data of the statement
		return List.of();
	}

	private static List<Map<String, Object>> resolveParameters(
			CqnUpdate statement, CdsEntity target, Iterable<Map<String, Object>> parameterValues) {

		List<Map<String, Object>> result;
		if (statement.entries().size() == 1) {
			result = new ArrayList<>();
			StreamSupport.stream(parameterValues.spliterator(), false).forEach(parameters ->
					result.add(mergeParametersAndSetters(target, parameters,
							statement.setters(), DataUtils.copyMap(statement.entries().get(0)))));
		} else {
			result = new ArrayList<>(statement.entries().size());
			Iterator<Map<String, Object>> iterator = parameterValues.iterator();
			statement.entries().forEach(r -> {
				Map<String, Object> parameters = iterator.next();
				if (parameters == null) {
					throw new IllegalStateException("""
							Unexpected condition: the size of the parameterValues \
							does not match the size of the entries in the statement\
							""");
				}
				result.add(mergeParametersAndSetters(target, parameters, statement.setters(), DataUtils.copyMap(r)));
			});
		}
		return result;
	}

	@SuppressWarnings("UnstableApiUsage")
	private static Map<String, Object> mergeParametersAndSetters(
			CdsEntity target, Map<String, Object> parameters, Map<String, CqnValue> setters, Map<String, Object> entry) {

		Map<String, Object> mutableParameters = new HashMap<>(parameters);

		setters.forEach((key, value) -> {
			if (value instanceof CqnParameter parameter) {
				mutableParameters.remove(parameter.name());
			} else if (value instanceof CqnExpression expression) {
				expression.accept(new CqnVisitor() {
					@Override
					public void visit(CqnParameter param) {
						mutableParameters.remove(param.name());
					}
				});
			}
			DataUtils.putPath(entry, key, null);
		});

		DataProcessor.create()
				.addConverter(
						(path, element, type) -> true,
						(path, element, value) -> element.isKey() ? value : null)
				.process(entry, target);

		// Parameters might define the values of the keys (they might not be present in the payload)
		entry.putAll(mutableParameters);
		return entry;
	}
}
