/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.identity;

import java.util.Optional;
import java.util.function.BiConsumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.services.runtime.CdsRuntimeConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.utils.IdentityUtils;
import com.sap.cds.services.utils.IdentityUtils.AuthenticationType;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;


public class IdentityConfiguration implements CdsRuntimeConfiguration {

	private final static Logger logger = LoggerFactory.getLogger(IdentityConfiguration.class);

	@Override
	public void eventHandlers(CdsRuntimeConfigurer configurer) {
		identityConfigured(configurer, (iasBinding, xsuaaBinding) -> configurer.eventHandler(new IdentityTenantProviderHandler(iasBinding, xsuaaBinding)));
	}

	@Override
	public void providers(CdsRuntimeConfigurer configurer) {
		identityConfigured(configurer, (iasBinding, xsuaaBinding) -> {
			logger.info("Loaded feature 'IdentityUserInfoProvider' (IAS: {}, XSUAA: {})",
					iasBinding.flatMap(ServiceBinding::getName).orElse("<none>"), xsuaaBinding.flatMap(ServiceBinding::getName).orElse("<none>"));
			configurer.provider(new IdentityUserInfoProvider(configurer.getCdsRuntime(), iasBinding, xsuaaBinding));
		});
	}

	private void identityConfigured(CdsRuntimeConfigurer configurer,
									BiConsumer<Optional<ServiceBinding>, Optional<ServiceBinding>> configure) {
		IdentityUtils identityUtils = new IdentityUtils(configurer.getCdsRuntime());

		AuthenticationType authType = identityUtils.configuredAuthenticationType();
		if (authType != null) {
			switch(authType) {
				case IAS:
					configure.accept(identityUtils.getIasServiceBindings().stream().findFirst(), Optional.empty());
					break;
				case IAS_XSUAA:
					configure.accept(identityUtils.getIasServiceBindings().stream().findFirst(), identityUtils.getXsuaaServiceBindings().stream().findFirst());
					break;
				case XSUAA:
					configure.accept(Optional.empty(), identityUtils.getXsuaaServiceBindings().stream().findFirst());
					break;
			}
		}
	}

}
