/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import java.sql.Connection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.CdsDataStore;
import com.sap.cds.CdsDataStoreConnector;
import com.sap.cds.CdsException;
import com.sap.cds.ConstraintViolationException;
import com.sap.cds.Result;
import com.sap.cds.impl.JDBCDataStoreConnector;
import com.sap.cds.ql.CdsDataException;
import com.sap.cds.ql.cqn.CqnValidationException;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.services.EventContext;
import com.sap.cds.services.cds.CdsCreateEventContext;
import com.sap.cds.services.cds.CdsDeleteEventContext;
import com.sap.cds.services.cds.CdsReadEventContext;
import com.sap.cds.services.cds.CdsUpdateEventContext;
import com.sap.cds.services.cds.CdsUpsertEventContext;
import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.changeset.ChangeSetContextSPI;
import com.sap.cds.services.changeset.ChangeSetListener;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.transaction.ChangeSetMemberDelegate;
import com.sap.cds.services.transaction.TransactionManager;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.services.utils.SessionContextUtils;
import com.sap.cds.services.utils.TenantAwareCache;
import com.sap.cds.services.utils.services.AbstractCdsService;

public class JdbcPersistenceService extends AbstractCdsService implements PersistenceService {

	private final static Logger logger = LoggerFactory.getLogger(JdbcPersistenceService.class);

	private final TenantAwareCache<CdsDataStoreConnector, CdsModel> cachedConnectors;
	private final Map<ChangeSetContext, CdsDataStore> cachedDataStores = new ConcurrentHashMap<>();
	private final TransactionManager txMgr;

	public JdbcPersistenceService(String name, Supplier<Connection> connectionSupplier, TransactionManager txMgr, CdsRuntime runtime) {
		super(name, runtime);
		this.txMgr = txMgr;
		this.cachedConnectors = TenantAwareCache.create(() -> new JDBCDataStoreConnector(
				RequestContext.getCurrent(runtime).getModel(), connectionSupplier, new com.sap.cds.transaction.TransactionManager() {
					@Override
					public boolean isActive() {
						ChangeSetContextSPI changeSetContext = (ChangeSetContextSPI) ChangeSetContext.getCurrent();
						boolean activeChangeSet = changeSetContext != null
								&& changeSetContext.hasChangeSetMember(txMgr.getName());
						boolean activeTxMgr = txMgr.isActive();
						return activeChangeSet || activeTxMgr;
					}

					@Override
					public void setRollbackOnly() {
						ChangeSetContext changeSetContext = ChangeSetContext.getCurrent();
						if (changeSetContext != null) {
							// this directly goes to the ChangeSet, instead of using the transaction manager
							changeSetContext.markForCancel();
						} else {
							txMgr.setRollbackOnly();
						}
					}
				}), runtime);
	}

	@Before
	@HandlerOrder(OrderConstants.Before.TRANSACTION_BEGIN)
	protected void ensureTransaction(EventContext context) {
		// register and begin transaction
		ChangeSetContextSPI changeSetContext = (ChangeSetContextSPI) context.getChangeSetContext();
		if (!changeSetContext.hasChangeSetMember(txMgr.getName())) {
			changeSetContext.register(new ChangeSetMemberDelegate(txMgr));
			txMgr.begin();
		}

		try {
			getCdsDataStore().setSessionContext(SessionContextUtils.toSessionContext(context));
		} catch (CdsException e) {
			logger.warn("Not supported to set the locale on the connection session", e);
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultRead(CdsReadEventContext context) {
		return checkExceptions(() -> getCdsDataStore().execute(context.getCqn(), context.getCqnNamedValues()));
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultCreate(CdsCreateEventContext context) {
		return checkExceptions(() -> getCdsDataStore().execute(context.getCqn()));
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultUpsert(CdsUpsertEventContext context) {
		return checkExceptions(() -> getCdsDataStore().execute(context.getCqn()));
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultUpdate(CdsUpdateEventContext context) {
		return checkExceptions(() -> getCdsDataStore().execute(context.getCqn(), context.getCqnValueSets()));
	}

	@On
	@HandlerOrder(OrderConstants.On.DEFAULT_ON)
	protected Result defaultDelete(CdsDeleteEventContext context) {
		return checkExceptions(() -> getCdsDataStore().execute(context.getCqn(), context.getCqnValueSets()));
	}

	@Override
	public CdsDataStore getCdsDataStore() {
		if(!ChangeSetContext.isActive()) {
			return cachedConnectors.findOrCreate().connect();
		}

		ChangeSetContext context = ChangeSetContext.getCurrent();
		CdsDataStore cdsDataStore = cachedDataStores.get(context);
		if(cdsDataStore == null) {
			cdsDataStore = cachedConnectors.findOrCreate().connect();
			context.register(new ChangeSetListener(){
				@Override
				public void afterClose(boolean completed) {
					cachedDataStores.remove(context);
				}
			});
			cachedDataStores.put(context, cdsDataStore);
		}

		return cdsDataStore;
	}

	private Result checkExceptions(Supplier<Result> supplier) {
		try {
			return supplier.get();
		} catch (ConstraintViolationException e) {
			throw new ErrorStatusException(CdsErrorStatuses.CONSTRAINT_VIOLATED, e.getMessage(), e); // TODO CDS4J support required
		} catch (CqnValidationException | CdsDataException e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_CQN, e.getMessage(), e); // TODO CDS4J support required
		}
	}

}
