/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.impl.persistence;

import java.sql.Connection;
import java.util.function.Supplier;

import javax.sql.DataSource;

import com.sap.cds.services.impl.transaction.LocalTransactionManager;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntimeConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.transaction.TransactionManager;
import com.sap.cds.transaction.impl.LocalTransactionManagerFactory;

public class JdbcPersistenceServiceConfiguration implements CdsRuntimeConfiguration {

	public static JdbcPersistenceServiceConfiguration create(DataSource dataSource) {
		return create(PersistenceService.DEFAULT_NAME, dataSource);
	}

	public static JdbcPersistenceServiceConfiguration create(String name, DataSource dataSource) {
		LocalTransactionManagerFactory txMgrFactory = new LocalTransactionManagerFactory(dataSource);
		return create(name, txMgrFactory.getConnectionSupplier(), new LocalTransactionManager(name, txMgrFactory.getTransactionManger()));
	}

	public static JdbcPersistenceServiceConfiguration create(String name, Supplier<Connection> connectionSupplier, TransactionManager txMgr) {
		return new JdbcPersistenceServiceConfiguration(name, connectionSupplier, txMgr);
	}

	private final String name;
	private final Supplier<Connection> connectionSupplier;
	private final TransactionManager txMgr;
	private JdbcPersistenceService service;

	private JdbcPersistenceServiceConfiguration(String name, Supplier<Connection> connectionSupplier, TransactionManager txMgr) {
		this.name = name;
		this.connectionSupplier = connectionSupplier;
		this.txMgr = txMgr;
	}

	public JdbcPersistenceService createOrGetService(CdsRuntimeConfigurer configurer) {
		if(service == null) {
			service = new JdbcPersistenceService(name, connectionSupplier, txMgr, configurer.getCdsRuntime());
			configurer.service(service);
		}
		return service;
	}

	@Override
	public void services(CdsRuntimeConfigurer configurer) {
		createOrGetService(configurer);
	}

}
