/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.feature.k8s;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.services.environment.CdsProperties.Environment.K8s;
import com.sap.cds.services.environment.ServiceBindingProvider;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cloud.environment.servicebinding.api.DefaultServiceBinding;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;

/**
 * The K8S based {@link ServiceBindingProvider}.
 * Providing service bindings based on K8S secrets mounted into the file system.
 */
public class K8sServiceBindingProvider implements ServiceBindingProvider {

	private static final ObjectMapper mapper = new ObjectMapper();
	private static final Logger logger = LoggerFactory.getLogger(K8sServiceBindingProvider.class);

	private final K8s config;
	private final File secretRoot;
	private List<ServiceBinding> serviceBindings;

	public K8sServiceBindingProvider(K8s config, File secretRoot) {
		this.config = config;
		this.secretRoot = secretRoot;
	}

	@Override
	public Stream<ServiceBinding> get() {
		if (serviceBindings == null) {
			List<ServiceBinding> theServiceBindings = new ArrayList<>();

			// get all services with naming convention <secretRoot>/service_name/instance_name/...
			if (secretRoot != null && secretRoot.isDirectory()) {
				Arrays.stream(secretRoot.listFiles())
						.filter(File::isDirectory)
						.map(this::findServiceInstances)
						.forEach(theServiceBindings::addAll);
			}

			// add service bindings which have been configured explicitly
			for(K8s.ServiceBindingConfig info : config.getServiceBindings().values()) {
				File serviceInstanceDirectory = new File(info.getSecretsPath());
				if (serviceInstanceDirectory.isDirectory()) {
					theServiceBindings.add(DefaultServiceBinding.builder().copy(Collections.emptyMap())
							.withName(info.getName()).withServiceName(info.getService()).withServicePlan(info.getPlan())
							.withTags(info.getTags() != null ? info.getTags() : Collections.emptyList())
							.withCredentials(getCredentials(serviceInstanceDirectory))
							.build());
				} else {
					throw new ErrorStatusException(CdsErrorStatuses.INVALID_SECRETS_PATH, info.getSecretsPath(), info.getName());
				}
			}

			serviceBindings = theServiceBindings;
		}
		return serviceBindings.stream();
	}

	private List<ServiceBinding> findServiceInstances(File serviceDirectory) {
		return Arrays.stream(serviceDirectory.listFiles()).filter(File::isDirectory).map(instanceDirectory -> {
			logger.debug("Found binding for service '{}' in instance directory '{}'", serviceDirectory.getName(), instanceDirectory.getAbsolutePath());

			return DefaultServiceBinding.builder().copy(Collections.emptyMap()).withName(instanceDirectory.getName())
					.withServiceName(serviceDirectory.getName()).withCredentials(getCredentials(instanceDirectory))
					.build();
		}).collect(Collectors.toList());
	}

	private Map<String, Object> getCredentials(File serviceInstanceDirectory) {
		Map<String, Object> credentials = new HashMap<>();
		Arrays.stream(serviceInstanceDirectory.listFiles())
				.filter(File::isFile)
				.filter(propertyFile -> !propertyFile.getName().startsWith("."))
				.forEach(propertyFile -> credentials.put(propertyFile.getName(), getValue(propertyFile)));
		return credentials;
	}

	private Object getValue(File propertyFile) {
		String content;
		try {
			content = new String(Files.readAllBytes(propertyFile.toPath()), StandardCharsets.UTF_8).trim();
		} catch (IOException e) {
			throw new ErrorStatusException(CdsErrorStatuses.INVALID_PROPERTY_FILE, propertyFile.getAbsolutePath(), e);
		}

		if(content.startsWith("[") || content.startsWith("{")) {
			try {
				return mapper.readValue(content, new TypeReference<Object>() {});
			} catch (JsonProcessingException e) { // NOSONAR
				// simple string, not a JSON object
			}
		}

		return content;
	}
}
