/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.mt.impl;

import com.sap.cds.feature.mt.MtUtils;
import com.sap.cds.feature.mt.SaasClient;
import com.sap.cds.feature.mt.SmsClient;
import com.sap.cds.services.auditlog.AuditLogService;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.mt.ExtensibilityService;
import com.sap.cds.services.mt.MtSubscriptionService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.utils.model.DynamicModelUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import com.sap.cloud.mt.subscription.InstanceLifecycleManager;

@SuppressWarnings("deprecation")
public class MtDeploymentServiceConfiguration implements CdsRuntimeConfiguration {

	@Override
	public void services(CdsRuntimeConfigurer configurer) {
		boolean compatibility = configurer.getCdsRuntime().getEnvironment().getCdsProperties().getMultiTenancy().getCompatibility().isEnabled();
		if (new MtUtils(configurer.getCdsRuntime()).requiresSubscription() && compatibility) {
			configurer.service(new MtSubscriptionServiceImpl(MtSubscriptionService.DEFAULT_NAME));
		}
		// isDynamicModelEnabled() doesn't fully guarantee extensibility usage (e.g. might only be feature toggles usage)
		if (new DynamicModelUtils(configurer.getCdsRuntime()).isDynamicModelEnabled()) {
			configurer.service(new ExtensibilityServiceImpl(ExtensibilityService.DEFAULT_NAME));
		}
	}

	@Override
	public void eventHandlers(CdsRuntimeConfigurer configurer) {
		CdsRuntime cdsRuntime = configurer.getCdsRuntime();

		MtUtils mtUtils = new MtUtils(cdsRuntime);
		InstanceLifecycleManager ilm = mtUtils.createDefaultInstanceLifecycleManager();

		ServiceBinding saasBinding = SaasClient.findBinding(cdsRuntime).orElse(null);
		ServiceBinding smsBinding = SmsClient.findBinding(cdsRuntime).orElse(null);

		if (saasBinding != null || smsBinding != null) {
			configurer.eventHandler(MtTenantProviderSaasOrSmsHandler.create(saasBinding, smsBinding));
		}

		if (ilm != null) {
			// adding this provider as well. SaaS handler is taken with priority based on the handler order
			configurer.eventHandler(new MtTenantProviderHandler(ilm));
		}

		if (mtUtils.requiresSubscription()) {
			boolean compatibility = cdsRuntime.getEnvironment().getCdsProperties().getMultiTenancy().getCompatibility().isEnabled();
			if (compatibility) {
				configurer.eventHandler(new MtSubscriptionServiceCompatibilityHandler(ilm, cdsRuntime));
			} else {
				configurer.eventHandler(new MtDeploymentServiceHandler(ilm, cdsRuntime));
			}

			// inform outbox about mt enablement, as persistent outbox is not usable for provider tenant on tenant-aware databases
			cdsRuntime.getEnvironment().getCdsProperties().getOutbox().getProviderTenant().setEnabled(false);
		}

		// Audit log tenant on/off boarding
		CdsProperties.AuditLog.StandardEvents standardEvents = configurer.getCdsRuntime().getEnvironment().getCdsProperties().getAuditLog().getStandardEvents();
		if (standardEvents.isEnabled() && standardEvents.getTenantLifecycleEvents().isEnabled()) {
			AuditLogService als = configurer.getCdsRuntime().getServiceCatalog().getService(AuditLogService.class,
					AuditLogService.DEFAULT_NAME);
			configurer.eventHandler(new TenantSubscriptionAuditLogHandler(als));
		}
	}
}
