/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.mt.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.Struct;
import com.sap.cds.feature.mt.SaasClient;
import com.sap.cds.feature.mt.SmsClient;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.mt.ReadProviderTenantEventContext;
import com.sap.cds.services.mt.ReadTenantsEventContext;
import com.sap.cds.services.mt.TenantInfo;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;

/**
 * Fetches tenant infos from both SaaS Registry and Subscription Manager
 * Service.
 */
@ServiceName(value = TenantProviderService.DEFAULT_NAME, type = TenantProviderService.class)
public class MtTenantProviderSaasOrSmsHandler implements EventHandler {

	private static final String KEY_TENANT_ID = "consumerTenantId";
	private static final String KEY_PROVIDER_TENANT_ID = "tenantid";
	public static final String SAAS = "saasRegistry";
	public static final String SUBSCRIBER = "subscriber";
	public static final String ZONE_ID = "zoneId";
	public static final String APP_TID = "app_tid";

	private final SaasClient saasClient;
	private final SmsClient smsClient;

	private final String providerTenant;

	@VisibleForTesting
	MtTenantProviderSaasOrSmsHandler(SaasClient saasClient, SmsClient smsClient, String providerTenant) {
		this.saasClient = saasClient;
		this.smsClient = smsClient;
		this.providerTenant = providerTenant;
	}

	public static MtTenantProviderSaasOrSmsHandler create(ServiceBinding saasBinding, ServiceBinding smsBinding) {
		SaasClient saasClient = saasBinding != null ? new SaasClient(saasBinding) : null;
		SmsClient smsClient = smsBinding != null ? new SmsClient(smsBinding) : null;

		String providerTenant = null;
		if (saasBinding != null) {
			providerTenant = (String) saasBinding.getCredentials().get(KEY_PROVIDER_TENANT_ID);
		} else if (smsBinding != null) {
			providerTenant = (String) smsBinding.getCredentials().get(KEY_PROVIDER_TENANT_ID);
		}
		return new MtTenantProviderSaasOrSmsHandler(saasClient, smsClient, providerTenant);
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	public void readTenants(ReadTenantsEventContext context) {
		try {
			// Get all SaaS TenantInfo objects by tenant
			Map<String, TenantInfo> tenantInfoByTenantFromSaas = new HashMap<>();
			if (saasClient != null) {
				saasClient.getSubscribedTenants().stream().map(this::toSaasTenantInfo)
						.forEach(t -> tenantInfoByTenantFromSaas.put(t.getTenant(), t));
			}

			// Add all SMS tenants and corresponding SaaS tenants
			List<TenantInfo> tenants = new ArrayList<>();
			if (smsClient != null) {
				smsClient.getSubscribedTenants().stream().map(this::toSmsTenantInfo).forEach(t -> {
					TenantInfo tenantInfoFromSaas = tenantInfoByTenantFromSaas.remove(t.getTenant());
					if (tenantInfoFromSaas != null) {
						t.put(SAAS, tenantInfoFromSaas);
					}
					tenants.add(t);
				});
			}

			// Add remaining SaaS tenants
			tenants.addAll(tenantInfoByTenantFromSaas.values());

			context.setResult(tenants);
		} catch (Exception e) {
			throw new ErrorStatusException(CdsErrorStatuses.TENANT_READ_FAILED, e);
		}
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	public void readProviderTenant(ReadProviderTenantEventContext context) {
		if (providerTenant != null) {
			context.setResult(providerTenant);
		}
	}

	private TenantInfo toSaasTenantInfo(Map<String, Object> tenant) {
		TenantInfo info = Struct.access(tenant).as(TenantInfo.class);
		info.setTenant(info.get(KEY_TENANT_ID).toString());
		return info;
	}

	@SuppressWarnings("unchecked")
	private TenantInfo toSmsTenantInfo(Map<String, Object> tenant) {
		TenantInfo info = Struct.access(tenant).as(TenantInfo.class);
		Object subscriber = info.get(SUBSCRIBER);
		String tenantId = (String) ((Map<String, Object>) subscriber).get(APP_TID);
		// fallback to zoneId if app_tid does not exist, yet
		if (tenantId == null) {
			tenantId = (String) ((Map<String, Object>) subscriber).get(ZONE_ID);
		}
		info.setTenant(tenantId);
		return info;
	}
}
