package com.sap.cds.feature.mt;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.BooleanNode;
import com.google.common.base.Objects;
import com.sap.cds.integration.cloudsdk.rest.client.JsonRestClient;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.environment.ServiceBindingUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import com.sap.cloud.environment.servicebinding.api.ServiceIdentifier;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultOAuth2PropertySupplier;
import com.sap.cloud.sdk.cloudplatform.connectivity.OAuth2ServiceBindingDestinationLoader;
import com.sap.cloud.sdk.cloudplatform.connectivity.OnBehalfOf;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationOptions;


/**
 * Implementation of the SaaS Registry REST client.
 */
public class SaasClient extends JsonRestClient {

	private static final Logger logger = LoggerFactory.getLogger(SaasClient.class);

	// could be configurable via CdsProperties
	private static final int PAGE_SIZE = 500;

	private static final String TENANT_STATE = "state";
	private static final String SAAS_REGISTRY = "saas-registry";

	private static final String SAAS_MANAGER_BASE = "/saas-manager/v1/";
	private static final String APPLICATION_TENANTS_PATH = SAAS_MANAGER_BASE + "application/subscriptions";
	private static final String SERVICE_TENANTS_PATH = SAAS_MANAGER_BASE + "service/subscriptions?includeIndirectSubscriptions=true";

	private final String urlPath;

	static {
		OAuth2ServiceBindingDestinationLoader.registerPropertySupplier(
			ServiceIdentifier.of(SAAS_REGISTRY),
			OAuth2Supplier::new);
	}

	private static class OAuth2Supplier extends DefaultOAuth2PropertySupplier {

		public OAuth2Supplier(ServiceBindingDestinationOptions options) {
			super(options, Collections.emptyList());
		}

		@Override
		public URI getServiceUri() {
			return getCredentialOrThrow(URI.class, "saas_registry_url");
		}

	}

	public static Optional<ServiceBinding> findBinding(CdsRuntime runtime) {
		return runtime.getEnvironment().getServiceBindings().filter(b -> ServiceBindingUtils.matches(b, null, SAAS_REGISTRY)).findFirst();
	}

	public SaasClient(ServiceBinding binding) {
		super(ServiceBindingDestinationOptions.forService(binding).onBehalfOf(OnBehalfOf.TECHNICAL_USER_PROVIDER).build());
		this.urlPath = "service".equals(binding.getServicePlan().orElse(null)) ? SERVICE_TENANTS_PATH : APPLICATION_TENANTS_PATH;
	}

	/**
	 * Retrieves the tenant info of all subscribed tenants
	 *
	 * @return tenants info
	 *
	 * @throws IOException throws when any connection problems occur
	 */
	public List<Map<String, Object>> getSubscribedTenants() throws IOException {

		logger.debug("Retrieving all tenants metadata from the SaaS registry");

		List<Map<String, Object>> result = new ArrayList<>();
		boolean morePages = true;
		int page = 1;

		while (morePages) {
			logger.debug("Retrieving page {} of all tenants metadata from the SaaS registry", page);

			String paginatedUrlPath = paginatedUrl(urlPath, page);
			JsonNode jsonResponse = getRequest(paginatedUrlPath);
			ArrayNode resp = getNode(jsonResponse, "subscriptions");
			BooleanNode morePagesResp = getNode(jsonResponse, "morePages");

			resp.forEach(tenant -> {
				String state = tenant.get(TENANT_STATE) != null ? tenant.get(TENANT_STATE).asText() : null;
				if (Objects.equal(state, "SUBSCRIBED") || Objects.equal(state, "UPDATE_FAILED")) {
					result.add(mapper.convertValue(tenant, new TypeReference<Map<String, Object>>() {
					}));
				}
			});

			morePages = morePagesResp != null && morePagesResp.booleanValue();
			page = page + 1;

			logger.debug("More tenants metadata from the SaaS registry available: {}", morePages ? "yes" : "no");
		}

		return result;
	}

	private String paginatedUrl(String urlPath, int page) {
		String url = urlPath + (urlPath.contains("?") ? "&" : "?");

		url = url + "page=" + page + "&size=" + PAGE_SIZE;

		return url;
	}

	@SuppressWarnings("unchecked")
	private  <N> N getNode(JsonNode jsonResponse, String keyName) {
		return (N) jsonResponse.get(keyName);
	}

}
