/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.mt;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.sap.cds.CdsData;
import com.sap.cds.Struct;

/**
 * This interface provides named and type safe access to the {@link HashMap options} provided by
 * {@link UnsubscribeEventContext#getOptions() }.
 */
public interface SaasRegistryUnsubscriptionOptions extends CdsData {

	/**
	 * Creates a new {@link SaasRegistryUnsubscriptionOptions}
	 * @return the {@link SaasRegistryUnsubscriptionOptions}
	 */
	static SaasRegistryUnsubscriptionOptions create() {
		return Struct.create(SaasRegistryUnsubscriptionOptions.class);
	}

	/**
	 * Creates a {@link SaasRegistryUnsubscriptionOptions} view over an existing options map.
	 * @param options the options map
	 * @return the {@link SaasRegistryUnsubscriptionOptions} view
	 */
	static SaasRegistryUnsubscriptionOptions create(Map<String, Object> options) {
		return Struct.access(options).as(SaasRegistryUnsubscriptionOptions.class);
	}

	/**
	 * Unique identifier of the subscription.
	 *
	 * @return unique identifier of the subscription.
	 */
	String getSubscriptionGUID();

	/**
	 * The application ID of the main subscribed application. Generated by XSUAA based on the xsappname.
	 *
	 * @return the application ID of the main subscribed application.
	 */
	String getSubscriptionAppId();

	/**
	 * The name of the main subscribed application, in the SaaS Provisioning.
	 *
	 * @return the name of the main subscribed application, in the SaaS Provisioning.
	 */
	String getSubscriptionAppName();

	/**
	 * The ID of the subscribed consumer XSUAA tenant.
	 *
	 * @return the ID of the subscribed consumer XSUAA tenant.
	 */
	String getSubscribedTenantId();

	/**
	 * The ID of the subscribed consumer zone. This field might be missing for subaccounts that were created before the
	 * Zone ID was introduced.
	 *
	 * @return The ID of the subscribed consumer zone.
	 */
	String getSubscribedZoneId();

	/**
	 * The License Type of the subscription consumer tenant. Values: UNDEFINED | CONSUMER | DEVELOPER | CUSTOMER |
	 * PARTNER | SAPDEV | SAPPROD | CUSTOMER_BETA | SYSTEM | SFSF_EXT_EVAL | TRIAL.
	 *
	 * @return The License Type of the subscription consumer tenant.
	 */
	String getSubscribedLicenseType();

	/**
	 * The CRM ID of the customer whose global account and subaccount triggered the subscription.
	 *
	 * @return The CRM ID of the customer whose global account and subaccount triggered the subscription.
	 */
	String getSubscribedCrmId();

	/**
	 * The Subdomain of the subscription consumer tenant (hostname for the identityzone).
	 *
	 * @return The Subdomain of the subscription consumer tenant (hostname for the identityzone).
	 */
	String getSubscribedSubdomain();

	/**
	 * The ID of the subaccount related to the consumer tenant.
	 *
	 * @return The ID of the subaccount related to the consumer tenant.
	 */
	String getSubscribedSubaccountId();

	/**
	 * The plan of the main subscribed application, as defined in CIS. If the app is not registered in the CIS catalog,
	 * or if the subscription flow is part of the "Stack Services", the subscriptionAppPlan would be null.
	 *
	 * @return The plan of the main subscribed application or null.
	 */
	String getSubscriptionAppPlan();

	/**
	 * The Amount that the customer is entitled to, in CIS.
	 *
	 * @return The Amount that the customer is entitled to.
	 */
	Integer getSubscriptionAppAmount();

	/**
	 * The List of App IDs, generated and returned by XSUAA when creating a subscription of a service instance (aka
	 * cloning). The App ID is a combination of some prefix and client ID of the subscribed service instance. This is
	 * relevant only for the PaaS Multitenancy scenario (aka stack services), therefore this list will be empty for SaaS
	 * applications.
	 *
	 * @return The List of App IDs, generated and returned by XSUAA when creating a subscription of a service instance.
	 */
	List<String> getDependentServiceInstanceAppIds();

	/**
	 * The ID of the subscribed consumer global account.
	 *
	 * @return The ID of the subscribed consumer global account.
	 */
	String getGlobalAccountGUID();

	/**
	 * The subaccount ID from which this app is provided.
	 *
	 * @return the subaccount ID from which this app is provided.
	 */
	String getProviderSubaccountId();

	/**
	 * The ID of the user who triggered the subscription from the cockpit (can be a clientId in case of client
	 * credentials flow).
	 *
	 * @return The ID of the user who triggered the subscription from the cockpit.
	 */
	String getUserId();

	/**
	 * This section may contain additional information on the user that triggered the unsubscription. This field is only
	 * relevant for accounts with "feature set B". If you use an account with "feature set A" this field does not exist.
	 *
	 * @return additional information on the user that triggered the unsubscription or null.
	 */
	SaasRegistryTriggeredByInfo getUserInfo();

	/**
	 * The parameters entered by the user upon subscription to your app.
	 *
	 * @return The parameters entered by the user upon subscription to your app.
	 */
	Map<Object, Object> getSubscriptionParams();

}
