/*******************************************************************************
 *   © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cds.feature.mt.lib.runtime;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Supplier;
import java.util.stream.Stream;

public class DataPoolSettings {

	private final Map<ConnectionPoolType, ParameterList> parameterLists = new EnumMap<>(ConnectionPoolType.class);

	/**
	 * Constructor
	 *
	 * @param prefixToPoolType maps a parameter prefix to a {@link ConnectionPoolType}
	 */
	public DataPoolSettings(Map<String, ConnectionPoolType> prefixToPoolType) {
		for (ConnectionPoolType type : ConnectionPoolType.values()) {
			parameterLists.put(type, type.createParameterList());
		}
		for (Map.Entry<String, ConnectionPoolType> entry : prefixToPoolType.entrySet()) {
			String prefix = entry.getKey();
			parameterLists.get(entry.getValue()).initList(prefix);
		}
	}

	public Stream<Parameter> getHikariParameters() {
		return parameterLists.get(ConnectionPoolType.HIKARI).stream();
	}

	public Stream<Parameter> getTomcatParameters() {
		return parameterLists.get(ConnectionPoolType.TOMCAT).stream();
	}

	public static class Parameter {
		private final String name;
		private final Class<?> type;
		private final String prefix;

		private Parameter(String name, Class<?> type, String prefix) {
			this.name = name;
			this.type = type;
			this.prefix = prefix;
		}

		public String getNameInEnv() {
			return (prefix.isBlank() ? "" : prefix + ".") + name;
		}

		public String getNormalizedNameInEnv() {
			return (prefix.isBlank() ? "" : prefix + ".") + normalizedName(name);
		}

		public String getPoolPropertyName() {
			return name;
		}

		public Class<?> getType() {
			return type;
		}

		public String getSetterName() {
			return "set" + name.substring(0, 1).toUpperCase(Locale.ENGLISH) + name.substring(1);
		}

		public String getPrefix() {
			return prefix;
		}
	}

	public enum ConnectionPoolType {
		HIKARI(HikariParameterList::new),
		TOMCAT(TomcatParameterList::new);

		private final Supplier<ParameterList> paramListSupplier;

		private ConnectionPoolType(Supplier<ParameterList> paramListSupplier) {
			this.paramListSupplier = paramListSupplier;
		}

		ParameterList createParameterList() {
			return paramListSupplier.get();
		}
	}

	public interface ParameterList extends List<Parameter> {

		void initList(String prefix);

	}

	@SuppressWarnings("serial")
	private static class HikariParameterList extends LinkedList<Parameter> implements ParameterList {

		@Override
		public void initList(String prefix) {
			var forbidden = List.of(
					"jdbcUrl",
					"schema",
					"dataSourceClassName",
					"poolName",
					"password",
					"username",
					"dataSource",
					"dataSourceJNDI",
					"driverClassName");
			generateParameterList(prefix, "com.zaxxer.hikari.HikariConfig")
					.stream().filter(p -> !forbidden.contains(p.getPoolPropertyName()))
					.forEach(this::add);
		}
	}

	@SuppressWarnings("serial")
	private static class TomcatParameterList extends LinkedList<Parameter> implements ParameterList {

		@Override
		public void initList(String prefix) {
			var forbidden = List.of("username", "password", "name", "url", "dataSource", "dataSourceJNDI",
					"driverClassName");
			generateParameterList(prefix, "org.apache.tomcat.jdbc.pool.PoolConfiguration")
					.stream().filter(p -> !forbidden.contains(p.getPoolPropertyName()))
					.forEach(this::add);
		}
	}

	private static List<Parameter> generateParameterList(String prefix, String className) {
		try {
			var clazz = Class.forName(className);
			Method[] methods = clazz.getDeclaredMethods();
			List<Parameter> parameters = new ArrayList<>();
			Arrays.stream(methods).filter(method -> Modifier.isPublic(method.getModifiers())).forEach(method -> {
				var paramTypes = method.getParameterTypes();
				if (paramTypes.length == 1) {
					if (method.getName().startsWith("set")) {
						parameters.add(new Parameter(getParameterName(method.getName().substring(3)), paramTypes[0], prefix));
					}
					if (method.getName().startsWith("is")) {
						parameters.add(new Parameter(getParameterName(method.getName().substring(2)), paramTypes[0], prefix));
					}
				}
			});
			return parameters;
		} catch (ClassNotFoundException e) {
			return new ArrayList<>();
		}
	}

	private static String getParameterName(String parameterName) {
		var firstChar = parameterName.length() > 0 ? Character.toLowerCase(parameterName.charAt(0)) : "";
		var rest = parameterName.length() > 1 ? parameterName.substring(1) : "";
		return firstChar + rest;
	}

	public static String normalizedName(String name) {
		if (name == null) {
			return "";
		}
		StringBuilder stringBuilder = new StringBuilder();
		for (int i = 0; i < name.length(); i++) {
			char current = name.charAt(i);
			if (Character.isUpperCase(current)) {
				if (i != 0) {
					stringBuilder.append("-");
				}
				stringBuilder.append(Character.toLowerCase(current));
			} else {
				stringBuilder.append(current);
			}
		}
		return stringBuilder.toString();
	}
}
