/*******************************************************************************
 *   © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cds.feature.mt.lib.runtime;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Stream;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.feature.mt.lib.runtime.DataPoolSettings.Parameter;
import com.sap.cds.feature.mt.lib.subscription.DataSourceInfo;
import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;
import com.sap.cds.feature.mt.lib.subscription.exceptions.MtLibError;

public abstract class DataSourceCreator {

	private static final Logger logger = LoggerFactory.getLogger(DataSourceCreator.class);

	public final DataSource create(DataSourceInfo info, EnvironmentAccess env) throws InternalError {
		DataSource dataSource = build(info);
		setPoolParameters(dataSource, env);
		return dataSource;
	}

	@SuppressWarnings("unchecked")
	private void setPoolParameters(DataSource dataSource, EnvironmentAccess env) {
		getPoolParameters().forEach(p -> {
			Object para;
			String normalizedNameInEnv = p.getNormalizedNameInEnv();
			if (p.getType() != Properties.class) {
				para = env.getProperty(normalizedNameInEnv, p.getType());
			} else {
				para = env.getPropertiesForPrefix(normalizedNameInEnv);
			}
			if (para != null) {
				try {
					if (p.getType() != Properties.class) {
						getMethod(dataSource, p).invoke(dataSource, para);
					} else {
						Properties properties = new Properties();
						properties.putAll((Map<Object, Object>) para);
						getMethod(dataSource, p).invoke(dataSource, properties);
					}
				} catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
					throw new MtLibError(e);
				} catch (IllegalArgumentException e) {
					throw new MtLibError(String.format("Invalid value for parameter '%s': %s (%s), expected value of type (%s)",
							normalizedNameInEnv, para, para.getClass().getCanonicalName(), p.getType().getCanonicalName()), e);
				}
			}
		});
	}

	protected abstract DataSource build(DataSourceInfo info) throws InternalError;

	protected abstract Stream<Parameter> getPoolParameters();

	private Method getMethod(DataSource dataSource, Parameter p) throws NoSuchMethodException {
		try {
			return dataSource.getClass().getMethod(p.getSetterName(), p.getType());
		} catch (NoSuchMethodException e) {
			logger.debug("Try with another type");
			if (p.getType() == boolean.class) {
				return dataSource.getClass().getMethod(p.getSetterName(), Boolean.class);
			} else if (p.getType() == int.class) {
				return dataSource.getClass().getMethod(p.getSetterName(), Integer.class);
			} else if (p.getType() == long.class) {
				return dataSource.getClass().getMethod(p.getSetterName(), Long.class);
			} else
				throw new NoSuchMethodException();
		}
	}
}
