/*******************************************************************************
 *   © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/

package com.sap.cds.feature.mt.lib.runtime;

import com.sap.cds.feature.mt.lib.subscription.InstanceLifecycleManager;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

public class DataSourceLookupBuilder {
	private static final String TOMCAT = "tomcat";
	private static final String HIKARI = "hikari";
	private InstanceLifecycleManager instanceLifecycleManager;
	private EnvironmentAccess environmentAccess;
	private String poolProvider;
	private boolean oneDataSourcePerDb;
	private boolean registerJmx;
	private Map<String, DataPoolSettings.ConnectionPoolType> prefixToPoolType;

	private DataSourceLookupBuilder() {
	}

	public static DataSourceLookupBuilder create() {
		return new DataSourceLookupBuilder();
	}

	public DataSourceLookup build() {
		DataPoolSettings dataPoolSettings;
		if (prefixToPoolType != null && !prefixToPoolType.isEmpty()) {
			dataPoolSettings = new DataPoolSettings(prefixToPoolType);
		} else {
			Map<String, DataPoolSettings.ConnectionPoolType> config = new HashMap<>();
			config.put("spring.datasource.hikari", DataPoolSettings.ConnectionPoolType.HIKARI);
			config.put("spring.datasource.tomcat", DataPoolSettings.ConnectionPoolType.TOMCAT);
			dataPoolSettings = new DataPoolSettings(config);
		}
		if (poolProvider == null || poolProvider.isEmpty()) {
			poolProvider = HIKARI;
		}
		switch (poolProvider.toLowerCase(Locale.ENGLISH)) {
			case TOMCAT:
				return new TomcatDataSourceLookup(instanceLifecycleManager, environmentAccess, dataPoolSettings,
						oneDataSourcePerDb, registerJmx);
			case HIKARI:
				return new HikariDataSourceLookup(instanceLifecycleManager, environmentAccess, dataPoolSettings,
						oneDataSourcePerDb);
			default:
				throw new IllegalArgumentException("Pool provider " + poolProvider + " is not supported");
		}
	}

	public DataSourceLookupBuilder instanceLifecycleManager(InstanceLifecycleManager instanceLifecycleManager) {
		this.instanceLifecycleManager = instanceLifecycleManager;
		return this;
	}

	public DataSourceLookupBuilder environmentAccess(EnvironmentAccess environmentAccess) {
		this.environmentAccess = environmentAccess;
		return this;
	}

	public DataSourceLookupBuilder poolProvider(String poolProvider) {
		this.poolProvider = poolProvider;
		return this;
	}

	public DataSourceLookupBuilder oneDataSourcePerDb(boolean oneDataSourcePerDb) {
		this.oneDataSourcePerDb = oneDataSourcePerDb;
		return this;
	}

	public DataSourceLookupBuilder registerJmx(boolean registerJmx) {
		this.registerJmx = registerJmx;
		return this;
	}

	public DataSourceLookupBuilder prefixToPoolType(Map<String, DataPoolSettings.ConnectionPoolType> prefixToPoolType) {
		this.prefixToPoolType = prefixToPoolType;
		return this;
	}
}
