/*******************************************************************************
 *   © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/

package com.sap.cds.feature.mt.lib.subscription;

import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;
import com.sap.cds.feature.mt.lib.subscription.json.SaasCallbackPayload;
import com.sap.cds.services.utils.lib.tools.api.ServiceEndpoint;
import com.sap.cds.services.utils.lib.tools.exception.InternalException;
import com.sap.cds.services.utils.lib.tools.exception.ServiceException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;

import static org.apache.http.HttpStatus.SC_BAD_GATEWAY;
import static org.apache.http.HttpStatus.SC_GATEWAY_TIMEOUT;
import static org.apache.http.HttpStatus.SC_INTERNAL_SERVER_ERROR;
import static org.apache.http.HttpStatus.SC_SERVICE_UNAVAILABLE;

public class SaasRegistry {
	public static final String SUCCEEDED = "SUCCEEDED";
	public static final String FAILED = "FAILED";
	public static final String SAAS_REGISTRY_DESTINATION = "com.sap.cds.saasRegistry";
	private static final String APPLICATION_JSON = "application/json";
	private static final Logger logger = LoggerFactory.getLogger(SaasRegistry.class);
	private ServiceSpecification serviceSpecification;

	public SaasRegistry(ServiceSpecification serviceSpecification) {
		this.serviceSpecification = serviceSpecification;
	}

	public void callBackSaasRegistry(boolean success, String message, String subscriptionUrl, String callbackUrl) throws InternalError {
		ServiceEndpoint saasRegistry = null;
		try {
			saasRegistry = ServiceEndpoint.create()
					.destinationName(SAAS_REGISTRY_DESTINATION).path(callbackUrl)
					.returnCodeChecker(c -> {
						if (c != HttpStatus.SC_OK) {
							return new InternalError("Saas registry returned http status " + c);
						} else {
							return null;
						}
					})
					.retry()
					.forReturnCodes(SC_BAD_GATEWAY, SC_GATEWAY_TIMEOUT, SC_INTERNAL_SERVER_ERROR, SC_SERVICE_UNAVAILABLE)
					.config(serviceSpecification.getResilienceConfig())
					.end();
		} catch (InternalException e) {
			throw new InternalError(e);
		}
		SaasCallbackPayload payload = getSaasCallbackPayload(success, message, subscriptionUrl);
		Map<String, String> headerFields = new HashMap<>();
		headerFields.put(HttpHeaders.ACCEPT, APPLICATION_JSON);
		logger.debug("Call saas registry");
		try {
			saasRegistry.createServiceCall().http().put().payload(payload).noPathParameter().noQuery()
					.enhancer(serviceSpecification.getRequestEnhancer())
					.insertHeaderFields(headerFields).end().execute();
		} catch (InternalException e) {
			throw new InternalError(e.getCause());
		} catch (ServiceException e) {
			if (e.getCause() instanceof InternalError internalError) {
				throw internalError;
			}
			throw new InternalError(e.getCause());
		}
	}

	private SaasCallbackPayload getSaasCallbackPayload(boolean success, String message, String subscriptionUrl) {
		SaasCallbackPayload payload = new SaasCallbackPayload();
		payload.message = message;
		if (success) {
			payload.status = SUCCEEDED;
		} else {
			payload.status = FAILED;
		}
		payload.subscriptionUrl = subscriptionUrl;
		return payload;
	}

	public ServiceSpecification getServiceSpecification() {
		return serviceSpecification;
	}
}
