/*******************************************************************************
 *   © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cds.feature.mt.lib.subscription;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.apache.commons.lang3.StringUtils;

import com.sap.cds.feature.mt.lib.subscription.ServiceOperation.Status;
import com.sap.cds.feature.mt.lib.subscription.ServiceOperation.Type;

/**
 * Service binding parameters
 */
public class ServiceBinding extends HashMap<String, Object> {
	public static final String CREDENTIALS = "credentials";
	public static final String SERVICE_INSTANCE_ID = "service_instance_id";
	public static final String ID = "id";
	public static final String NAME = "name";
	private final ServiceOperation lastOperation;

	public ServiceBinding(Map<String, ?> m) {
		super(m);
		lastOperation = new ServiceOperation(m != null ? m : new HashMap<String, Object>());
	}

	public String getId() {
		return containsKey(ID) ? (String) get(ID) : "";
	}

	public boolean isReady() {
		return containsKey("ready") && (boolean) get("ready");
	}

	public String getName() {
		return containsKey(NAME) ? (String) get(NAME) : "";
	}

	public String getServiceInstanceId() {
		return containsKey(SERVICE_INSTANCE_ID) ? (String) get(SERVICE_INSTANCE_ID) : "";
	}

	public Instant getCreatedAt() {
		return Instant.parse((String) get("created_at"));
	}

	public Instant getUpdatedAt() {
		return Instant.parse((String) get("updated_at"));
	}

	@SuppressWarnings("unchecked")
	public Map<String, Object> getCredentials() {
		return containsKey(CREDENTIALS) ? (Map<String, Object>) get(CREDENTIALS) : new HashMap<>();
	}

	public ServiceOperation getLastOperation() {
		return lastOperation;
	}

	@SuppressWarnings("unchecked")
	public List<String> getTenants() {
		var labels = (Map<String, Object>) get("labels");
		if (labels == null) {
			return new ArrayList<>();
		}
		var tenants = (List<String>) labels.get("tenant_id");
		return tenants != null ? tenants.stream().filter(t -> !StringUtils.isBlank(t)).toList() : new ArrayList<>();
	}

	public boolean hasTenant() {
		return !getTenants().isEmpty();
	}

	boolean isUsable() {
		var status = lastOperation.getStatus();
		return isReady() &&
				!getCredentials().isEmpty() &&
				!(lastOperation.getType() == Type.CREATE && (status == Status.FAILED || status == Status.IN_PROGRESS));
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;
		if (!super.equals(o)) return false;
		ServiceBinding that = (ServiceBinding) o;
		return Objects.equals(lastOperation, that.lastOperation);
	}

	@Override
	public int hashCode() {
		return Objects.hash(super.hashCode(), lastOperation);
	}
}
