/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.mt.impl;

import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.feature.mt.lib.subscription.Subscriber;
import com.sap.cds.feature.mt.lib.subscription.exceptions.AuthorityError;
import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;
import com.sap.cds.feature.mt.lib.subscription.exceptions.NotFound;
import com.sap.cds.feature.mt.lib.subscription.exceptions.NotSupported;
import com.sap.cds.feature.mt.lib.subscription.exceptions.ParameterError;

public class AsyncSidecarUpgradeHelper {

	private static final Logger logger = LoggerFactory.getLogger(AsyncSidecarUpgradeHelper.class);
	private static final ObjectMapper mapper = new ObjectMapper()
			.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
	private static final String SIDECAR_STATUS_RUNNING = "RUNNING";
	private static final String SIDECAR_STATUS_FAILED = "FAILED";

	private final Subscriber subscriber;

	public AsyncSidecarUpgradeHelper(Subscriber subscriber) {
		this.subscriber = subscriber;
	}

	public void deploy(List<String> tenants) throws AuthorityError, ParameterError, InternalError, NotSupported, NotFound {
		try {
			String jobIdJson = subscriber.setupDbTablesAsync(tenants);
			String jobId = mapper.readValue(jobIdJson, JobIdResult.class).jobID;

			String statusJson;
			SidecarResult sidecarResult;
			do {
				Thread.sleep(2000);
				statusJson = subscriber.updateStatus(jobId);
				sidecarResult = mapper.readValue(statusJson, SidecarResult.class);
			} while (SIDECAR_STATUS_RUNNING.equals(upper(sidecarResult.status)));

			boolean failed = SIDECAR_STATUS_FAILED.equals(upper(sidecarResult.status));
			if (!failed && sidecarResult.tenants != null) { // streamlined MTX
				failed = sidecarResult.tenants.values().stream().anyMatch(tenant -> SIDECAR_STATUS_FAILED.equals(upper(tenant.status)));
			}

			if (failed) {
				throw new InternalError("Updating failed with response '" + statusJson + "'");
			}
			logger.debug("Updating succeeded with response '{}'", statusJson);
		} catch (JsonProcessingException e) {
			throw new InternalError("Failed to parse sidecar response", e);
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}

	}

	private static final class SidecarResult {
		public String status;
		public Map<String, TenantResult> tenants;

		private static final class TenantResult {
			public String status;
		}
	}

	private static final class JobIdResult {
		public String jobID;
	}

	private String upper(String s) {
		return s == null ? s : s.toUpperCase(Locale.ENGLISH);
	}

}
