/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.mt.impl;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.mt.ReadTenantsEventContext;
import com.sap.cds.services.mt.TenantInfo;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cds.feature.mt.lib.subscription.InstanceLifecycleManager;
import com.sap.cds.feature.mt.lib.subscription.TenantMetadata;
import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;

@ServiceName(value = TenantProviderService.DEFAULT_NAME, type = TenantProviderService.class)
public class MtTenantProviderHandler implements EventHandler {

	private static final String ENRICH_TENANT = "enrich_tenant";

	private final InstanceLifecycleManager ilm;

	public MtTenantProviderHandler(InstanceLifecycleManager ilm) {
		this.ilm = ilm;
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE + 1)
	public void readTenants(ReadTenantsEventContext context) {
		try {
			context.setResult(ilm.getAllTenantInfos(true).values().stream()
					.map(tenant -> toTenantInfo(tenant, context.getFields())).collect(Collectors.toList()));
			context.put(ENRICH_TENANT, false);
		} catch (InternalError e) {
			throw new ErrorStatusException(CdsErrorStatuses.TENANT_READ_FAILED, e);
		}
	}

	/**
	 * {@link After} handler which is primarily used to enrich an existing list of tenants that has
	 * been resolved before (e.g. by {@link MtTenantProviderSaasOrSmsHandler} with additional fields
	 * (e.g. 'database_id'). The enrichment is skipped in case the event context contains a field
	 * {@link #ENRICH_TENANT}. This might be the case if the required information has been added
	 * already before.
	 *
	 * @param context {@link ReadTenantsEventContext} which already contains a list of tenants
	 */
	@After
	public void enrichTenants(ReadTenantsEventContext context) {
		Object enrichTenant = context.get(ENRICH_TENANT);
		Set<String> fields = context.getFields();
		if (Boolean.FALSE.equals(enrichTenant)
				|| (fields != null && !fields.isEmpty() && !fields.contains(InstanceLifecycleManager.DATABASE_ID))) {
			return;
		}
		try {
			Map<String, TenantMetadata> smTenants = ilm.getAllTenantInfos(false);
			// data needs to be merged into the original list as this might already contain additional properties like 'subdomain'
			List<TenantInfo> tenantInfo = context.getResult();
			tenantInfo.forEach(tenant -> {
				if(smTenants.containsKey(tenant.getTenant())) {
					addDatabaseId(smTenants.get(tenant.getTenant()), tenant);
				}
			});
		} catch (InternalError e) {
			throw new ErrorStatusException(CdsErrorStatuses.TENANT_READ_FAILED, e);
		}
	}

	private static void addDatabaseId(TenantMetadata  smTenant, TenantInfo tenant) {
		Map<String, Object> props = smTenant.getAdditionalProperties();
		if(props!=null && props.containsKey(InstanceLifecycleManager.DATABASE_ID)) {
			tenant.put(InstanceLifecycleManager.DATABASE_ID, props.get(InstanceLifecycleManager.DATABASE_ID));
		}
	}

	private TenantInfo toTenantInfo(TenantMetadata tenant, Set<String> fields) {
		TenantInfo info = TenantInfo.create();
		if (fields == null || fields.isEmpty() || fields.contains(TenantInfo.TENANT)) {
			info.setTenant(tenant.getTenantId());
		}
		if (fields == null || fields.isEmpty() || fields.contains(InstanceLifecycleManager.DATABASE_ID)) {
			addDatabaseId(tenant, info);
		}
		return info;
	}

}
