/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.mt.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.UnaryOperator;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.feature.mt.SaasClient;
import com.sap.cds.feature.mt.SmsClient;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.HandlerOrder;
import com.sap.cds.services.handler.annotations.On;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.mt.ReadProviderTenantEventContext;
import com.sap.cds.services.mt.ReadTenantsEventContext;
import com.sap.cds.services.mt.TenantInfo;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.OrderConstants;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;

/**
 * Fetches tenant infos from both SaaS Registry and Subscription Manager
 * Service.
 */
@ServiceName(value = TenantProviderService.DEFAULT_NAME, type = TenantProviderService.class)
public class MtTenantProviderSaasOrSmsHandler implements EventHandler {

	private static final String KEY_PROVIDER_TENANT_ID = "tenantid";
	public static final String SAAS = "saasRegistry";

	private final SaasClient saasClient;
	private final SmsClient smsClient;

	private final String providerTenant;

	@VisibleForTesting
	MtTenantProviderSaasOrSmsHandler(SaasClient saasClient, SmsClient smsClient, String providerTenant) {
		this.saasClient = saasClient;
		this.smsClient = smsClient;
		this.providerTenant = providerTenant;
	}

	public static MtTenantProviderSaasOrSmsHandler create(ServiceBinding saasBinding, ServiceBinding smsBinding) {
		SaasClient saasClient = saasBinding != null ? new SaasClient(saasBinding) : null;
		SmsClient smsClient = smsBinding != null ? new SmsClient(smsBinding) : null;

		String providerTenant = null;
		if (saasBinding != null) {
			providerTenant = (String) saasBinding.getCredentials().get(KEY_PROVIDER_TENANT_ID);
		} else if (smsBinding != null) {
			providerTenant = (String) smsBinding.getCredentials().get(KEY_PROVIDER_TENANT_ID);
		}
		return new MtTenantProviderSaasOrSmsHandler(saasClient, smsClient, providerTenant);
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	public void readTenants(ReadTenantsEventContext context) {
		try {
			// Get all SaaS TenantInfo objects by tenant
			Map<String, TenantInfo> tenantInfoByTenantFromSaas = new HashMap<>();

			Set<String> filter = context.getFields();
			UnaryOperator<TenantInfo> filterFunction = getFieldsFilter(filter);

			if (saasClient != null) {
				saasClient.getSubscribedTenants(filterFunction).forEach(t -> tenantInfoByTenantFromSaas.put(t.getTenant(), t));
			}

			// Add all SMS tenants and corresponding SaaS tenants
			List<TenantInfo> tenants = new ArrayList<>();
			if (smsClient != null) {
				smsClient.getSubscribedTenants(filterFunction).forEach(t -> {
					TenantInfo tenantInfoFromSaas = tenantInfoByTenantFromSaas.remove(t.getTenant());
					if (tenantInfoFromSaas != null) {
						t.put(SAAS, tenantInfoFromSaas);
					}
					tenants.add(t);
				});
			}

			// Add remaining SaaS tenants
			tenants.addAll(tenantInfoByTenantFromSaas.values());

			context.setResult(tenants);
		} catch (Exception e) {
			throw new ErrorStatusException(CdsErrorStatuses.TENANT_READ_FAILED, e);
		}
	}

	protected static UnaryOperator<TenantInfo> getFieldsFilter(Set<String> filter) {
		UnaryOperator<TenantInfo> filterFunction = UnaryOperator.identity();
		if (filter != null && !filter.isEmpty()) {
			filterFunction = info -> {
				info.keySet().retainAll(filter);
				return info;
			};
		}
		return filterFunction;
	}

	@On
	@HandlerOrder(OrderConstants.On.FEATURE)
	public void readProviderTenant(ReadProviderTenantEventContext context) {
		if (providerTenant != null) {
			context.setResult(providerTenant);
		}
	}

}
