/*
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt;

import com.sap.cds.feature.mt.lib.runtime.DataPoolSettings;
import com.sap.cds.feature.mt.lib.runtime.DataPoolSettings.ConnectionPoolType;
import com.sap.cds.feature.mt.lib.runtime.DataSourceLookup;
import com.sap.cds.feature.mt.lib.runtime.DataSourceLookupBuilder;
import com.sap.cds.feature.mt.lib.runtime.EnvironmentAccess;
import com.sap.cds.feature.mt.lib.runtime.IdentityZoneDeterminer;
import com.sap.cds.feature.mt.lib.runtime.TenantAwareDataSource;
import com.sap.cds.feature.mt.lib.runtime.TenantProvider;
import com.sap.cds.feature.mt.lib.subscription.InstanceLifecycleManager;
import com.sap.cds.services.datasource.DataSourceFactory;
import com.sap.cds.services.environment.CdsProperties.MultiTenancy;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeAware;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.datasource.DataSourceUtils;
import com.sap.cds.services.utils.datasource.DataSourceUtils.PoolType;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import javax.sql.DataSource;

/** Provider of a tenant dependent data source. */
public class RoutingDataSourceFactory implements DataSourceFactory, CdsRuntimeAware {
  private static final String DATA_SOURCE_SECTION_MT = "cds.multitenancy.datasource.";

  private CdsRuntime runtime;
  private MtUtils mtUtils;

  @Override
  public void setCdsRuntime(CdsRuntime runtime) {
    this.runtime = runtime;
    this.mtUtils = new MtUtils(runtime);
  }

  @Override
  public Map<String, DataSource> create() {
    Map<String, DataSource> dataSources = new HashMap<>();
    for (Entry<String, InstanceLifecycleManager> entry :
        mtUtils.createInstanceLifecycleManagers().entrySet()) {
      dataSources.put(entry.getKey(), createRoutingDataSource(entry.getKey(), entry.getValue()));
    }
    return dataSources;
  }

  private DataSource createRoutingDataSource(String name, InstanceLifecycleManager ilm) {
    Map<String, DataPoolSettings.ConnectionPoolType> parameterConfig = createParameterConfig(name);
    MultiTenancy multiTenancy = runtime.getEnvironment().getCdsProperties().getMultiTenancy();
    DataSourceLookup lookup =
        DataSourceLookupBuilder.create()
            .instanceLifecycleManager(ilm)
            .environmentAccess(
                new EnvironmentAccess() {
                  @Override
                  public <T> T getProperty(String key, Class<T> clazz) {
                    return runtime.getEnvironment().getProperty(key, clazz, null);
                  }

                  @Override
                  @SuppressWarnings("unchecked")
                  public Map<Object, Object> getPropertiesForPrefix(String prefix) {
                    return getProperty(prefix, HashMap.class);
                  }
                })
            .prefixToPoolType(parameterConfig)
            .poolProvider(multiTenancy.getDataSource().getPool())
            .combinePools(multiTenancy.getDataSource().getCombinePools().isEnabled())
            .build();
    TenantProvider provider =
        new TenantProvider(
            new IdentityZoneDeterminer() {
              @Override
              public String getIdentityZone() throws InternalError {
                UserInfo userInfo = RequestContext.getCurrent(runtime).getUserInfo();
                DatabaseSchemaIdProvider providedMapper =
                    runtime.getProvider(DatabaseSchemaIdProvider.class);
                String tenant =
                    providedMapper != null
                        ? providedMapper.get(userInfo, name)
                        : userInfo.getTenant();
                if (tenant == null) {
                  throw new ErrorStatusException(CdsErrorStatuses.TENANT_CONTEXT_MISSING);
                }
                return tenant;
              }
            });
    return new TenantAwareDataSource(provider, lookup, runtime);
  }

  private Map<String, DataPoolSettings.ConnectionPoolType> createParameterConfig(String name) {
    Map<String, DataPoolSettings.ConnectionPoolType> config = new LinkedHashMap<>();
    addConfigParameter(config, ConnectionPoolType.HIKARI, PoolType.HIKARI, name);
    addConfigParameter(config, ConnectionPoolType.TOMCAT, PoolType.TOMCAT, name);
    return config;
  }

  private void addConfigParameter(
      Map<String, DataPoolSettings.ConnectionPoolType> config,
      DataPoolSettings.ConnectionPoolType type,
      PoolType poolType,
      String name) {
    config.put(DataSourceUtils.getDataSourceSection(name, poolType), type);
    config.put(DATA_SOURCE_SECTION_MT + poolType, type);
  }
}
