/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.feature.mt.lib.runtime;

import com.sap.cds.feature.mt.lib.runtime.DataPoolSettings.Parameter;
import com.sap.cds.feature.mt.lib.subscription.DataSourceInfo;
import com.sap.cds.feature.mt.lib.subscription.exceptions.InternalError;
import java.lang.management.ManagementFactory;
import java.util.stream.Stream;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.sql.DataSource;
import org.apache.tomcat.jdbc.pool.ConnectionPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataSourceCreatorTomcat extends DataSourceCreator {
  private static final Logger logger = LoggerFactory.getLogger(DataSourceCreatorTomcat.class);
  private final DataPoolSettings poolSettings;
  private final boolean registerJmx;

  public DataSourceCreatorTomcat(DataPoolSettings poolSettings, boolean registerJmx) {
    this.poolSettings = poolSettings;
    this.registerJmx = registerJmx;
  }

  @Override
  protected DataSource build(DataSourceInfo info) throws InternalError {
    org.apache.tomcat.jdbc.pool.DataSource dataSource =
        new org.apache.tomcat.jdbc.pool.DataSource();
    dataSource.setDriverClassName(info.getDriver());
    dataSource.setUrl(info.getUrl());
    dataSource.setUsername(info.getUser());
    dataSource.setPassword(info.getPassword());
    if (registerJmx) {
      MBeanServer server = ManagementFactory.getPlatformMBeanServer();
      try {
        dataSource.createPool();
        ObjectName objectName =
            new ObjectName(
                ConnectionPool.POOL_JMX_TYPE_PREFIX
                    + "ConnectionPool"
                    + ",class=org.apache.tomcat.jdbc.pool.DataSource"
                    + ",name="
                    + info.getTenantId());
        dataSource.preRegister(server, objectName);
      } catch (Exception e) {
        logger.error("Could not register JMX. Error is", e);
      }
    }
    return dataSource;
  }

  @Override
  protected Stream<Parameter> getPoolParameters() {
    return poolSettings.getTomcatParameters();
  }
}
